from commons.aws_helper import AWSHelper

from botocore.exceptions import ClientError

import time
import re
import json
import yaml
from cloudformation.utils import set_generic_constructor
from cloudformation.utils.eks_mapper import EKSMapper
from cloudformation.utils.s3_bucket_mapper import S3BucketMapper


class CFClient(object):
    def __init__(self, account, region):
        self.account = account
        self.region = region
        self.config = AWSHelper.create_config(self.account)
        self.session = AWSHelper.create_session(account, region_name=region)
        AWSHelper.set_ca_bundle_path()
        self.cf_client = self.get_client('cloudformation')
        self.resourceMappers = {
            'AWS::EKS::Cluster': EKSMapper(self.get_client('eks', False)),
            'AWS::S3::Bucket': S3BucketMapper(self.get_client('s3'))
        }

    def get_client(self, resource_name, verify_ssl = True):
        if not self.account.verifySSL or not verify_ssl:
            return self.session.client(resource_name, region_name=self.region, verify=False, config=self.config)
        return self.session.client(resource_name, region_name=self.region, config=self.config)

    @staticmethod
    def new_instance(container):
        set_generic_constructor()
        return CFClient(container.account, container.region)

    def validate_template(self, deployed):
        _, template = self.read_stack_file(deployed)
        try:
            return self.cf_client.validate_template(TemplateBody=template)
        except ClientError as err:
            print("Validation failed for stack [%s]. %s" % (deployed.name, err.message))
            raise

    def create_stack(self, deployed):
        stackname = self._sanatize_name(deployed.name)
        capabilities = deployed.capabilities
        if not self._stack_exists(stackname):
            parameters, template = self.read_stack_file(deployed)
            self.cf_client.create_stack(StackName=stackname, TemplateBody=template, Parameters=parameters,
                                        Capabilities=capabilities, DisableRollback=deployed.disableRollback)
            deployed.stackId = self._stack_exists(stackname)
            return True
        return False

    def update_stack(self, deployed):
        capabilities = deployed.capabilities
        stack_id = self._stack_exists(self._sanatize_name(deployed.name))
        if not stack_id:
            raise Exception("Stack '%s' does not exist." % deployed.name)
        else:
            deployed.stackId = stack_id

        parameters, template = self.read_stack_file(deployed)
        self.cf_client.update_stack(StackName=stack_id, TemplateBody=template, Parameters=parameters,
                                    Capabilities=capabilities)

    def read_stack_file(self, deployed):
        # get content of cf file
        with open(deployed.file.path, 'r') as tfile:
            template = tfile.read()
        parameters = []
        for k in deployed.inputVariables:
            param = {}
            param['ParameterKey'] = k
            param['ParameterValue'] = deployed.inputVariables[k]
            parameters.append(param)
        return parameters, template

    def describe_stack(self, deployed, stack_state="present"):
        stack_id = self._stack_exists(self._sanatize_name(deployed.name), stack_state)
        if stack_id:
            # I've seen a scenario where the stack exists as checked above
            # but describe_stacks fails with the stack not found.
            try:
                return self.cf_client.describe_stacks(StackName=stack_id)['Stacks'][0]
            except ClientError, arg:
                print
                "WARN: Describe Stack generated exception. The stack probably doesn't exist. '%s'" % arg
                return None

        return None

    def get_stack_events(self, deployed, stack_state="present", next_token=None):
        stackname = self._sanatize_name(deployed.name)
        if deployed.stackId:
            stack_id = deployed.stackId
        else:
            stack_id = stackname
        if self._stack_exists(stackname, stack_state):
            try:
                if next_token:
                    return self.cf_client.describe_stack_events(StackName=stack_id, NextToken=next_token)
                else:
                    return self.cf_client.describe_stack_events(StackName=stack_id)
            except ClientError, arg:
                print
                "WARN: Describe Stack Events generated exception.'%s'" % arg
                return None
        return None

    def list_resources(self, deployed):
        stackname = self._sanatize_name(deployed.name)
        if self._stack_exists(stackname):
            return self.cf_client.list_stack_resources(StackName=stackname)['StackResourceSummaries']
        return None

    def capture_output(self, deployed):
        stack_name = self._sanatize_name(deployed.name)
        if self._stack_exists(stack_name):
            stack = self.cf_client.describe_stacks(StackName=stack_name)['Stacks'][0]
            output_variables = {}
            if "Outputs" in stack:
                outputs = stack['Outputs']
                for output in outputs:
                    output_variables[output['OutputKey']] = output['OutputValue']
            deployed.outputVariables = output_variables
            return True

        return False

    def destroy_stack(self, deployed, wait=True, sleep_interval=5):
        stackname = self._sanatize_name(deployed.name)
        if self._stack_exists(stackname):
            self.cf_client.delete_stack(StackName=stackname)

            return True
        return False

    def wait_for_ready_status(self, deployed, context, sleep_interval=6):
        ready_statuses = ['CREATE_COMPLETE', 'UPDATE_COMPLETE']
        stopped_statuses = ['CREATE_FAILED', 'ROLLBACK_FAILED', 'ROLLBACK_COMPLETE', 'UPDATE_ROLLBACK_FAILED',
                            'UPDATE_ROLLBACK_COMPLETE']
        wait_statuses = ['CREATE_IN_PROGRESS', 'ROLLBACK_IN_PROGRESS', 'DELETE_IN_PROGRESS', 'UPDATE_IN_PROGRESS',
                         'UPDATE_COMPLETE_CLEANUP_IN_PROGRESS', 'UPDATE_ROLLBACK_COMPLETE_CLEANUP_IN_PROGRESS',
                         'UPDATE_ROLLBACK_IN_PROGRESS', 'REVIEW_IN_PROGRESS']
        max_intervals = deployed.timeoutInterval

        interval_cnt = 0
        while True:
            interval_cnt += 1
            if interval_cnt > max_intervals:
                raise Exception("Stack [%s] timed out waiting for 'Complete'" % (deployed.name))

            stack = self.describe_stack(deployed)
            if stack is None:
                time.sleep(sleep_interval)
                continue
            deployed.last_event_id = self.print_events(deployed, context)
            stack_status = stack['StackStatus']
            if stack_status in ready_statuses:
                return True
            elif stack_status in stopped_statuses:
                raise Exception("Expected stack [%s] to be 'Complete', but was '%s'" % (deployed.name, stack_status))
            elif stack_status in wait_statuses:
                time.sleep(sleep_interval)
            else:
                raise Exception("Unknown stack status '%s'" % stack_status)

    @staticmethod
    def get_string_event(event):
        event_data = "{}: {}".format(event['Timestamp'], event['ResourceStatus'])
        if 'LogicalResourceId' in event:
            event_data = "{} for {}".format(event_data, event['LogicalResourceId'])
        if 'ResourceStatusReason' in event:
            event_data = "{}. Reason: {}".format(event_data, event['ResourceStatusReason'])
        return event_data

    @staticmethod
    def get_string_resource(resource):
        resource_data = "\nType: {}".format(resource['ResourceType'])
        if 'LogicalResourceId' in resource:
            resource_data = "{}\nLogical Resource Id: {}".format(resource_data, resource['LogicalResourceId'])
        if 'PhysicalResourceId' in resource:
            resource_data = "{}\nPhysical Resource Id: {}".format(resource_data, resource['PhysicalResourceId'])
        if 'ResourceStatus' in resource:
            resource_data = "{}\nStatus: {}".format(resource_data, resource['ResourceStatus'])
        if 'LastUpdatedTimestamp' in resource:
            resource_data = "{}\nLast Updated: {}".format(resource_data, resource['LastUpdatedTimestamp'])
        return resource_data

    def print_resources(self, deployed):
        print("\nResources:")
        resources = self.list_resources(deployed)
        if resources:
            for resource in resources:
                print("\n------------------------{}".format(self.get_string_resource(resource)))
            print("\n------------------------")

    def wait_for_terminated_status(self, deployed, context, sleep_interval=15):
        stackname = self._sanatize_name(deployed.name)
        max_intervals = deployed.timeoutInterval

        if not self._stack_exists(stackname):
            return False

        stopped_statuses = ['DELETE_COMPLETE']
        failed_statuses = ['CREATE_FAILED', 'DELETE_FAILED', 'ROLLBACK_FAILED', 'ROLLBACK_COMPLETE',
                           'UPDATE_ROLLBACK_FAILED', 'UPDATE_ROLLBACK_COMPLETE']
        wait_statuses = ['CREATE_IN_PROGRESS', 'ROLLBACK_IN_PROGRESS', 'DELETE_IN_PROGRESS', 'UPDATE_IN_PROGRESS',
                         'UPDATE_COMPLETE_CLEANUP_IN_PROGRESS', 'UPDATE_ROLLBACK_COMPLETE_CLEANUP_IN_PROGRESS',
                         'UPDATE_ROLLBACK_IN_PROGRESS', 'REVIEW_IN_PROGRESS']

        interval_cnt = 0
        while True:
            interval_cnt += 1
            if interval_cnt > max_intervals:
                raise Exception("Stack [%s] timed out waiting for 'Delete'" % (deployed.name))

            stack = self.describe_stack(deployed, stack_state="terminate")
            deployed.last_event_id = self.print_events(deployed, context, stack_state="terminate")
            stack_status = stack['StackStatus']
            if stack_status in stopped_statuses:
                return True
            elif stack_status in failed_statuses:
                raise Exception("Stack termination failed with '%s'" % stack_status)
            elif stack_status in wait_statuses:
                time.sleep(sleep_interval)
            else:
                raise Exception("Unknown stack status '%s'" % stack_status)

    def print_events(self, deployed, context, stack_state="present"):
        last_event_id = deployed.last_event_id
        new_events = []
        has_more = True
        next_token = None
        while has_more:
            all_events = self.get_stack_events(deployed, stack_state, next_token)
            for event in all_events['StackEvents']:
                if last_event_id and event['EventId'] == last_event_id:
                    has_more = False
                    break
                else:
                    new_events.append(event)
            has_more = has_more and 'NextToken' in all_events and all_events['NextToken']
            if has_more:
                next_token = all_events['NextToken']
        if len(new_events):
            last_event_id = new_events[0]['EventId']
        for event in reversed(new_events):
            string_event = self.get_string_event(event)
            if 'FAILED' in event['ResourceStatus']:
                context.logError(string_event)
            else:
                context.logOutput(string_event)
        return last_event_id

    def get_template_body(self, deployed):
        stackname = self._sanatize_name(deployed.name)
        if not self._stack_exists(stackname):
            return None

        with open(deployed.file.path, 'r') as tfile:
            template = tfile.read()
        try:
            print("Loading Json")
            return json.loads(template)
        except:
            try:
                print("Loading YAML ")
                return yaml.load(template)
            except:
                raise

    # INTERNAL FUNCTIONS =========================================================

    # AWS requires that stack names match regex [a-zA-Z][-a-zA-Z0-9]*
    def _sanatize_name(self, name):
        p = re.compile('_| |\$|&|%|@|!|#')
        return p.sub('-', name)

        # Get a list of stacks and see if any match the given name.

    def _stack_exists(self, name, state="present"):
        retry = True
        next_token = None
        while retry:
            if next_token:
                response = self.cf_client.list_stacks(NextToken=next_token)
            else:
                response = self.cf_client.list_stacks()

            stack_id = self._get_stack_id(name, response['StackSummaries'], state)
            if stack_id:
                return stack_id
            else:
                if 'NextToken' in response:
                    next_token = response['NextToken']
                else:
                    retry = False

        return False

    def _get_stack_id(self, name, stacks, state):
        for stack in stacks:
            if 'StackName' in stack and stack['StackName'] != name:
                continue

            if 'StackStatus' not in stack or stack['StackStatus'] != 'DELETE_COMPLETE':
                # name matches and not deleted
                return stack['StackId']
            elif 'StackStatus' not in stack or (stack['StackStatus'] == 'DELETE_COMPLETE' and state == "terminate"):
                return stack['StackId']

        return False

    def create_cis(self, resource, cloud):
        if not 'ResourceType' in resource:
            raise Exception("No 'ResourceType' on resource")
        elif resource['ResourceType'] in self.resourceMappers:
            return self.resourceMappers[resource['ResourceType']].create_cis(resource, cloud)
        return None
