import io
import json
from collections import deque

import requests
from xld.azure.connector import AzureConnector
from requests.exceptions import RequestException

class AzurePyHelper():
    def __init__(self, cloud):
        self.cloud = cloud
        self.connector = connector = AzureConnector(cloud)
        self.proxies = None
        if connector.proxiesdict:
            self.proxies = json.dumps(self.connector.proxiesdict)

    def deploy_resource_group_template(self, deployment_name, resource_group_name, cf_template, parameters, deployment_mode):
        url = "https://management.azure.com/subscriptions/{}/resourcegroups/{}/providers/Microsoft.Resources/deployments/{}?api-version=2020-10-01".format(self.cloud.subscriptionId, resource_group_name, deployment_name)
        body_data = {
            "properties": {
                "mode": deployment_mode,
                "template": cf_template,
                "parameters": parameters
            }
        }
        json_body_data = json.dumps(body_data)
        headers = {'Content-Type': 'application/json','Authorization': 'Bearer ' + self.connector.credentials.token['access_token']}
        try:
            response = requests.put(url, data=json_body_data, headers=headers, proxies=self.connector.proxiesdict)
            if not response.ok:
                print response.status_code
                print response.reason
                raise Exception("Error submitting Template for deployment")

            logger.info("Template Deployment to ResourceGroup {0} submitted".format(resource_group_name))
            print "Template Deployment to ResourceGroup {0} submitted".format(resource_group_name)
        except RequestException as e:
            print(e)
            raise Exception("Error submitting Template for deployment", e)

    def get_deployment_state(self, deployment_name, resource_group_name):

        deployment_obj = self.get_deployment(deployment_name, resource_group_name)
        try:
            if deployment_obj:
                return deployment_obj['properties']['provisioningState']
        except KeyError:
            print "No deployment properties returned. Response below"
            print deployment_obj
            return None
        else:
            return None

    def delete_resource_group(self, resource_group_name):
        url = "https://management.azure.com/subscriptions/{}/resourcegroups/{}?api-version=2021-04-01".format(self.cloud.subscriptionId, resource_group_name)
        headers = {'Content-Type': 'application/json','Authorization': 'Bearer ' + self.connector.credentials.token['access_token']}
        try:
            response = requests.delete(url, headers=headers, proxies=self.connector.proxiesdict)
            logger.info("Resource is deleted: {0}".format(resource_group_name))
            print "Resource Group {0} deleted.".format(resource_group_name)
        except RequestException as e:
            print(e)
            raise Exception("Error deleting Resource Group", e)

    @staticmethod
    def loadTemplate(templatePath, file_encoding='UTF-8'):
        with io.open(templatePath, encoding=file_encoding) as f:
            content = f.read()
        return json.loads(str(content))

    def get_deployment(self, deployment_name, resource_group_name):
        url = "https://management.azure.com/subscriptions/{}/resourcegroups/{}/providers/Microsoft.Resources/deployments/{}?api-version=2021-04-01".format(self.cloud.subscriptionId, resource_group_name, deployment_name)
        headers = {'Content-Type': 'application/json','Authorization': 'Bearer ' + self.connector.credentials.token['access_token']}
        try:
            response = requests.get(url, headers=headers, proxies=self.connector.proxiesdict)
            logger.info("Deployment {0} details retrieved".format(deployment_name))
            print "Deployment {0} details retrieved".format(deployment_name)
        except RequestException as e:
            if response.status_code == 404:
                logger.info("Deployment {0} Not found yet".format(deployment_name))
                print "Deployment {0} Not found yet".format(deployment_name)
                return None
            print(e)
            logger.info("Error retrieving deployment {0} details".format(deployment_name))
            raise Exception("Error retrieving deployment details", e)

        response_obj = json.loads(response.content)
        return response_obj

    def get_failed_deployment_operations(self, deployment_name, resource_group_name):
        url = "https://management.azure.com/subscriptions/{}/resourcegroups/{}/deployments/{}/operations?api-version=2021-04-01".format(self.cloud.subscriptionId, resource_group_name, deployment_name)
        headers = {'Content-Type': 'application/json','Authorization': 'Bearer ' + self.connector.credentials.token['access_token']}

        try:
            response = requests.get(url, headers=headers, proxies=self.connector.proxiesdict)
            logger.info("Deployment {0} operations retrieved".format(deployment_name))
            print "Deployment {0} operations retrieved".format(deployment_name)
        except RequestException as e:
            print(e)
            logger.info("Error retrieving deployment {0} operations".format(deployment_name))
            raise Exception("Error retrieving deployment operations", e)

        operations = []
        try:
            response_obj = json.loads(response.content)
            operations_arr = response_obj['value']

            for operation in operations_arr:
                try:
                    if operation['properties']['statusCode'] is not 'OK':
                        operations.append("%s:%s" % (operation['properties']['statusCode'],operation['properties']['statusMessage']['error']['message']))
                except:
                    logger.info("Failed to extract deployment operation")
        except:
            logger.info("Error retrieving deployment {0} operations".format(deployment_name))
            print response_obj

        return operations

    def get_deployment_output(self, deployment_name, resource_group_name):
        deployment_obj = self.get_deployment(deployment_name, resource_group_name)
        if deployment_obj:
            try:
                return deployment_obj['properties']['outputs']
            except KeyError:
                print "No deployment properties returned. Response below"
                print deployment_obj
                return None
        else:
            return None

    def get_deployment_resources(self, deployment_name, resource_group_name):
        deployment_obj = self.get_deployment(deployment_name, resource_group_name)
        if deployment_obj:
            try:
                return deployment_obj['properties']['outputResources']
            except KeyError:
                print "No deployment properties returned. Response below"
                print deployment_obj
                return None
        else:
            return None

    def delete_resource(self, resource_id):
        logger.info("Deleting Resource %s" % resource_id)
        print "Deleting Resource " + resource_id

        url = "https://management.azure.com/{}?api-version=2021-04-01".format(resource_id)
        headers = {'Content-Type': 'application/json','Authorization': 'Bearer ' + self.connector.credentials.token['access_token']}

        try:
            response = requests.delete(url, headers=headers, proxies=self.connector.proxiesdict)
        except RequestException as e:
            print(e)
            logger.info("Error deleting resource {0}".format(resource_id))

        logger.info("Resource " + resource_id  + "deleted")
        print "Deleted Resource " + resource_id

    def get_cluster_admin_credentials(self, resource_group_name, resource_name):
        logger.info("Retrieving cluster admin cred")
        url = "https://management.azure.com/subscriptions/{}/resourceGroups/{}/providers/Microsoft.ContainerService/managedClusters/{}/listClusterAdminCredential?api-version=2023-01-01".format(self.cloud.subscriptionId, resource_group_name, resource_name)
        headers = {'Content-Type': 'application/json','Authorization': 'Bearer ' + self.connector.credentials.token['access_token']}

        try:
            response = requests.post(url, headers=headers, proxies=self.connector.proxiesdict)
            response_obj = json.loads(response.content)
            return response_obj['kubeconfigs'][0]['value']
        except RequestException as e:
            logger.info("Error Retrieving cluster admin cred")
            raise Exception("Error retrieving cluster admin cred", e)

        return None

    def get_resource(self, resource_id):
        logger.info("Retrieving resource details")
        url = "https://management.azure.com/{}?api-version=2021-04-01".format(resource_id)
        headers = {'Content-Type': 'application/json','Authorization': 'Bearer ' + self.connector.credentials.token['access_token']}

        try:
            response = requests.get(url, headers=headers, proxies=self.connector.proxiesdict)
            return json.loads(response.content)
        except RequestException as e:
            print(e)
            logger.info("Error retrieving resource details")

        return None

    def create_webapp(self, resource_group_name, webapp_name, data):
        logger.info("Creating webapp")
        url = "https://management.azure.com/subscriptions/{}/resourceGroups/{}/providers/Microsoft.Web/sites/{}?api-version=2022-03-01".format(self.cloud.subscriptionId, resource_group_name, webapp_name)
        headers = {'Content-Type': 'application/json','Authorization': 'Bearer ' + self.connector.credentials.token['access_token']}

        try:
            response = self.connector.httpRequestHelper.executeRequest(url,"PUT",json.dumps(headers),data,None)
            if not self.connector.httpRequestHelper.getStatusCode() < 300:
                raise Exception(("Error creating webapp %s" % webapp_name), response['response'])
        except Exception as e:
            print(e)
            logger.info(("Error creating webapp %s" % webapp_name))

        return None

    def delete_webapp(self, resource_group_name, webapp_name):
        logger.info("Deleting webapp")
        url = "https://management.azure.com/subscriptions/{}/resourceGroups/{}/providers/Microsoft.Web/sites/{}?deleteEmptyServerFarm=false&api-version=2022-03-01".format(self.cloud.subscriptionId, resource_group_name, webapp_name)
        headers = {'Content-Type': 'application/json','Authorization': 'Bearer ' + self.connector.credentials.token['access_token']}

        try:
            response = self.connector.httpRequestHelper.executeRequest(url,"DELETE",json.dumps(headers),None,None)
            if not self.connector.httpRequestHelper.getStatusCode() < 300:
                raise Exception(("Error deleting webapp %s" % webapp_name), response['response'])
        except Exception as e:
            print(e)
            logger.info(("Error deleting webapp %s" % webapp_name))

        return None

    def delete_webjob(self, url, web_job_name, headerBasic):
        logger.info("Deleting webjob")
        headers = {'Content-Type': 'application/json'}
        headerBasicDict = dict(headerBasic)
        headers.update(headerBasicDict)
        try:
            response = self.connector.httpRequestHelper.executeRequest(url,"DELETE",json.dumps(headers),None,None)
            if not self.connector.httpRequestHelper.getStatusCode() < 300:
                raise Exception(("Error deleting webjob %s" % web_job_name), response['response'])
        except Exception as e:
            print(e)
            logger.info(("Error deleting webjob %s" % web_job_name))

        return None

    def get_deployment_status(self, status_code):
        if status_code == 0:
            return "Undefined"
        elif status_code == 1:
            return "NotDeployed"
        elif status_code == 2:
            return "InProgress"
        elif status_code == 4:
            return "Succeeded"
        elif status_code == 8:
            return "PartiallySucceeded"
        elif status_code == 16:
            return "Failed"
        else:
            return "unknown deployment status %d" % status_code
