import tempfile
import urllib
import os
from azure.mgmt.resource import ResourceManagementClient
from azure.mgmt.storage import StorageManagementClient
from azure.mgmt.network import NetworkManagementClient
from azure.mgmt.compute import ComputeManagementClient
from azure.mgmt.web import WebSiteManagementClient
from azure.mgmt.containerinstance import ContainerInstanceManagementClient
from azure.keyvault import KeyVaultAuthentication
from azure.keyvault.key_vault_client import KeyVaultClient
from azure.mgmt.keyvault import KeyVaultManagementClient
from com.xebialabs.deployit.plugin.azure import LoaderUtil
from java.nio.file import Files, Paths, StandardCopyOption

from azure.common.credentials import ServicePrincipalCredentials
from azure.common.credentials import UserPassCredentials
from com.xebialabs.deployit.plugin.azure import HttpRequestHelper

import com.xebialabs.deployit.plugin.azure.TrustExtractHelper as TrustExtractHelper
from msrest.service_client import ServiceClient

import ssl
from requests.adapters import HTTPAdapter


class AzureConnector(object):
    proxiesdict = None
    def __init__(self, container):
        self.verify_ssl = container.verifySSL
        AzureConnector.set_ca_bundle_path()

        if not self.verify_ssl:
            self._disable_https_warnings()

        self.credentials = {}
        self.vaultcredentials = {}

        proxies = AzureConnector.create_proxy(container)
        self.proxiesdict = proxies
        self.client_proxies = ClientProxies()
        self.client_proxies.proxies = proxies

        port_proxy = 0
        if container.proxyPort is not None:
            port_proxy = container.proxyPort

        self.httpRequestHelper = HttpRequestHelper(container.proxyHost, port_proxy, container.proxyUser, container.proxyPassword, container.verifySSL)

        if container.authMethod == 'active-directory-user-password':
            if (not container.userName) or (not container.userPassword):
                raise Exception('For active-directory-user-password \
                                 authentication parameters userName \
                                 and userPassword are required')
            else:
                self.credentials = UserPassCredentials(
                    username=container.userName,
                    password=container.userPassword,
                    proxies=proxies,
                    verify=self.verify_ssl
                )
                self.vaultcredentials = UserPassCredentials(
                    username=container.userName,
                    password=container.userPassword,
                    proxies=proxies,
                    verify=self.verify_ssl
                )

        elif container.authMethod == 'directory-application-and-service-principal':
            if (not container.clientId) or (not container.clientKey) or (not container.tenantId):
                raise Exception('For directory-application-and-service-principal \
                                 authentication parameters clientId, clientKey,\
                                 tenantId are required')
            else:
                self.credentials = ServicePrincipalCredentials(
                    client_id=container.clientId,
                    secret=container.clientKey,
                    tenant=container.tenantId,
                    proxies=proxies,
                    verify=self.verify_ssl
                )
                self.vaultcredentials = ServicePrincipalCredentials(
                    client_id=container.clientId,
                    secret=container.clientKey,
                    tenant=container.tenantId,
                    proxies=proxies,
                    verify=self.verify_ssl
                )

        else:
            raise Exception('Authentication method {0} not supported'.format(container.authMethod))

        self.subscription_id = container.subscriptionId

    def resource_client(self):
        resource_client = ResourceManagementClient(self.credentials, str(self.subscription_id))
        return self.set_ssl_params(resource_client)

    def storage_client(self):
        storage_client = StorageManagementClient(self.credentials, str(self.subscription_id))
        return self.set_ssl_params(storage_client)

    def network_client(self):
        network_client = NetworkManagementClient(self.credentials, str(self.subscription_id))
        return self.set_ssl_params(network_client)

    def compute_client(self):
        compute_client = ComputeManagementClient(self.credentials, str(self.subscription_id))
        return self.set_ssl_params(compute_client)

    def container_client(self):
        container_client = ContainerInstanceManagementClient(self.credentials, str(self.subscription_id))
        return self.set_ssl_params(container_client)

    def web_client(self):
        web_client = WebSiteManagementClient(self.credentials, str(self.subscription_id))
        return self.set_ssl_params(web_client)

    def keyvault_management_client(self):
        keyvault_management_client = KeyVaultManagementClient(
            self.credentials, str(self.subscription_id))
        return self.set_ssl_params(keyvault_management_client)

    def keyvault_client(self):
        def _auth_callback(server, resource, scope):
            vault_credentials = self.vaultcredentials
            vault_credentials.verify = self.verify_ssl
            vault_credentials.resource = resource
            vault_credentials.set_token()
            return vault_credentials.scheme, vault_credentials.__dict__['token']['access_token']
        return KeyVaultClient(KeyVaultAuthentication(_auth_callback))

    def _disable_https_warnings(self):
        import requests.packages.urllib3
        requests.packages.urllib3.disable_warnings()

    def _extract_file_from_jar(self, config_file):
        file_url = LoaderUtil.getResourceBySelfClassLoader(config_file)
        if file_url:
            tmp_file, tmp_abs_path = tempfile.mkstemp()
            tmp_file.close()
            Files.copy(file_url.openStream(), Paths.get(tmp_abs_path), StandardCopyOption.REPLACE_EXISTING)
            return tmp_abs_path
        else:
            return None

    def resource_wait_for_delete(self, resource_group_name, resource_name, resource_type):
        result = ''
        resource_client = self.resource_client()
        is_group_present = resource_client.resource_groups.check_existence(
            resource_group_name=resource_group_name
        )

        if is_group_present:
            existing_resources = resource_client.resources.list_by_resource_group(
                resource_group_name=resource_group_name,
                filter="name eq '{0}' and resourceType eq '{1}'".format(resource_name, resource_type),
                verify=False
            )
            for item in existing_resources:
                print("Found resource %s, retrying" % item.id)
                result = "RETRY"

        return result

    @staticmethod
    def create_proxy(container):
        proxies = None
        if container.proxyHost and container.proxyProtocol and container.proxyPort:
            proxy = "{}:{}".format(container.proxyHost, container.proxyPort)
            if container.proxyUser and container.proxyPassword:
                proxy = "{}:{}@{}".format(urllib.quote(container.proxyUser), urllib.quote(container.proxyPassword), proxy)
            proxy = "{}://{}".format(container.proxyProtocol, proxy)
            proxies = {'https': proxy, 'http': proxy}
        return proxies

    @staticmethod
    def set_ca_bundle_path():
        os.environ['REQUESTS_CA_BUNDLE'] = TrustExtractHelper.getCACertPath()

    def set_ssl_params(self, client):
        client.config.connection.verify = self.verify_ssl
        if self.verify_ssl:
            client.config.session_configuration_callback = custom_session_configuration_callback

        if self.proxiesdict:
            config = client._client.config
            config.proxies = self.client_proxies
            client._client = ServiceClient(self.credentials,  config)
        return client

    def get_requests_session(self):
        from requests import Session
        session = Session()
        if self.verify_ssl:
            session.mount("https://", SSLContextAdapter())
        return session

def custom_session_configuration_callback(session, global_config, local_config, **kwargs):
    session.mount("https://", SSLContextAdapter())
    return kwargs

class SSLContextAdapter(HTTPAdapter):
    def init_poolmanager(self, *args, **kwargs):
        context=ssl.SSLContext(ssl.PROTOCOL_TLSv1_2)
        context.verify_mode = ssl.CERT_REQUIRED
        context.check_hostname = True
        kwargs['ssl_context'] = context
        return super(SSLContextAdapter, self).init_poolmanager(*args, **kwargs)

class ClientProxies(object):

    def __init__(self):
        self.proxies = {}
        self.use_env_settings = True

    def __call__(self):
        return self.proxies

    def add(self, protocol, proxy_url):
        self.proxies[protocol] = proxy_url
