import tempfile
from azure.mgmt.resource import ResourceManagementClient
from azure.mgmt.storage import StorageManagementClient
from azure.mgmt.network import NetworkManagementClient
from azure.mgmt.compute import ComputeManagementClient
from azure.mgmt.web import WebSiteManagementClient
from azure.mgmt.containerinstance import ContainerInstanceManagementClient
from azure.keyvault import KeyVaultAuthentication
from azure.keyvault.key_vault_client import KeyVaultClient
from azure.mgmt.keyvault import KeyVaultManagementClient
from com.xebialabs.deployit.plugin.azure import LoaderUtil
from java.nio.file import Files, Paths, StandardCopyOption

from azure.common.credentials import ServicePrincipalCredentials
from azure.common.credentials import UserPassCredentials


class AzureConnector(object):
    def __init__(self, container):
        self._disable_https_warnings()
        self._set_ca_bundle_path()

        self.credentials = {}
        self.vaultcredentials = {}

        if container.authMethod == 'active-directory-user-password':
            if (not container.userName) or (not container.userPassword):
                raise Exception('For active-directory-user-password \
                                 authentication parameters userName \
                                 and userPassword are required')
            else:
                self.credentials = UserPassCredentials(
                    username=container.userName,
                    password=container.userPassword,
                )
                self.vaultcredentials = UserPassCredentials(
                    username=container.userName,
                    password=container.userPassword,
                )

        elif container.authMethod == 'directory-application-and-service-principal':
            if (not container.clientId) or (not container.clientKey) or (not container.tenantId):
                raise Exception('For directory-application-and-service-principal \
                                 authentication parameters clientId, clientKey,\
                                 tenantId are required')
            else:
                self.credentials = ServicePrincipalCredentials(
                    client_id=container.clientId,
                    secret=container.clientKey,
                    tenant=container.tenantId
                )
                self.vaultcredentials = ServicePrincipalCredentials(
                    client_id=container.clientId,
                    secret=container.clientKey,
                    tenant=container.tenantId
                )

        else:
            raise Exception('Authentication method {0} not supported'.format(container.authMethod))

        self.subscription_id = container.subscriptionId

    def resource_client(self):
        resource_client = ResourceManagementClient(self.credentials, str(self.subscription_id))
        return resource_client

    def storage_client(self):
        storage_client = StorageManagementClient(self.credentials, str(self.subscription_id))
        return storage_client

    def network_client(self):
        network_client = NetworkManagementClient(self.credentials, str(self.subscription_id))
        return network_client

    def compute_client(self):
        compute_client = ComputeManagementClient(self.credentials, str(self.subscription_id))
        return compute_client

    def container_client(self):
        container_client = ContainerInstanceManagementClient(self.credentials, str(self.subscription_id))
        return container_client

    def web_client(self):
        web_client = WebSiteManagementClient(self.credentials, str(self.subscription_id))
        return web_client

    def keyvault_management_client(self):
        keyvault_management_client = KeyVaultManagementClient(
            self.credentials, str(self.subscription_id))
        return keyvault_management_client

    def keyvault_client(self):
        def _auth_callback(server, resource, scope):
            vault_credentials = self.vaultcredentials
            vault_credentials.resource = resource
            vault_credentials.set_token()
            return vault_credentials.scheme, vault_credentials.__dict__['token']['access_token']
        return KeyVaultClient(KeyVaultAuthentication(_auth_callback))

    def _disable_https_warnings(self):
        import requests.packages.urllib3
        requests.packages.urllib3.disable_warnings()

    def _set_ca_bundle_path(self):
        ca_bundle_path = self._extract_file_from_jar("certifi/cacert.pem")
        import os
        os.environ['REQUESTS_CA_BUNDLE'] = ca_bundle_path

    def _extract_file_from_jar(self, config_file):
        file_url = LoaderUtil.getResourceBySelfClassLoader(config_file)
        if file_url:
            tmp_file = None
            try:
                tmp_file, tmp_abs_path = tempfile.mkstemp()
                Files.copy(file_url.openStream(), Paths.get(tmp_abs_path), StandardCopyOption.REPLACE_EXISTING)
                return tmp_abs_path
            finally:
                if tmp_file:
                    tmp_file.close()
        else:
            return None

    def resource_wait_for_delete(self, resource_group_name, resource_name, resource_type):
        result = ''
        resource_client = self.resource_client()
        is_group_present = resource_client.resource_groups.check_existence(
            resource_group_name=resource_group_name
        )

        if is_group_present:
            existing_resources = resource_client.resources.list_by_resource_group(
                resource_group_name=resource_group_name,
                filter="name eq '{0}' and resourceType eq '{1}'".format(resource_name, resource_type),
                verify=False
            )
            for item in existing_resources:
                print "Found resource %s, retrying" % item.id
                result = "RETRY"

        return result
