from cf.client import SpaceClient
from org.cloudfoundry import AbstractCloudFoundryException
from cf.space.helpers.space_helper import SpaceHelper
from java.lang import IllegalArgumentException, IllegalStateException

class AppHelper(object):
    def __init__(self, container):
        self.space_client = SpaceClient.create_client(container)
        self.space_helper = SpaceHelper(container.container)

    @staticmethod
    def get_app_name(deployed):
        return deployed.appName if deployed.appName else deployed.name

    @staticmethod
    def get_service_instance_name(boundService):
        return boundService.serviceInstanceName if boundService.serviceInstanceName else boundService.name

    def create_app(self, deployed):
        print "Creating app {} on {}".format(self.get_app_name(deployed),
                                                     self.space_helper.get_space_name(deployed.container))
        try:
            self.space_client.create_application(self.get_app_name(deployed),
                                                 deployed.file.path, deployed.memory, deployed.instances,
                                                 self.get_build_pack_url(deployed),
                                                 deployed.hostname, deployed.randomRoute, deployed.noRoute)
        except AbstractCloudFoundryException as e:
            print "Error occurred during app creation: {}. Rolling back.".format(e.message)
            self.space_client.delete_application(self.get_app_name(deployed))
            raise e

    def get_app_details(self, app_name):
        return self.space_client.get_application_details(app_name=app_name)

    def delete_app(self, space_name, app_name):
        print "Destroying app {} from {}".format(app_name, space_name)
        self.space_client.delete_application(app_name)

    def modify_app(self, deployed):
        print "Updating app {} on {}".format(self.get_app_name(deployed),
                                                     self.space_helper.get_space_name(deployed.container))
        self.space_client.push_application(self.get_app_name(deployed),
                                           deployed.file.path, deployed.memory, deployed.instances,
                                           self.get_build_pack_url(deployed),
                                           deployed.hostname, deployed.randomRoute, deployed.noRoute)

    def start_app(self, app_name):
        print "Starting app {}".format(app_name)
        self.space_client.start_application(app_name)

    def stop_app(self, app_name):
        print "Stopping app {}".format(app_name)
        self.space_client.stop_application(app_name)

    def rename_app(self, app_name, new_app_name):
        print "Renaming app {} to {} ".format(app_name, new_app_name)
        self.space_client.rename_application(app_name, new_app_name)

    def bind_app_to_service(self, app_name, bound_service):
        print "Binding app {} to service {}".format(app_name, self.get_service_instance_name(bound_service))
        self.space_client.bind_app_to_service(app_name, self.get_service_instance_name(bound_service),
                                              bound_service.parameters)

    def unbind_app_from_service(self, app_name, bound_service):
        print "Unbinding app {} from service {}".format(app_name, self.get_service_instance_name(bound_service))
        try:
            self.space_client.unbind_app_from_service(app_name, self.get_service_instance_name(bound_service))
        except (IllegalArgumentException, IllegalStateException) as e:
            if "does not exist" in e.getMessage() or "not bound to application" in e.getMessage():
                print "Binding of service {} and app {} does not exist. Ignoring unbinding".format(self.get_service_instance_name(bound_service), app_name)
            else:
                raise e

    @staticmethod
    def get_build_pack_url(deployed):
        build_pack_repo_mapping = {"Binary": "https://github.com/cloudfoundry/binary-buildpack.git",
                                   "Go": "https://github.com/cloudfoundry/go-buildpack.git",
                                   "HWC": "https://github.com/cloudfoundry/hwc.git",
                                   "Java": "https://github.com/cloudfoundry/java-buildpack.git",
                                   ".Net": "https://github.com/cloudfoundry/dotnet-core-buildpack.git",
                                   "NGINX": "https://github.com/cloudfoundry/nginx-buildpack.git",
                                   "Node.js": "https://github.com/cloudfoundry/nodejs-buildpack.git",
                                   "PHP": "https://github.com/cloudfoundry/php-buildpack.git",
                                   "Python": "https://github.com/cloudfoundry/python-buildpack.git",
                                   "R": "https://github.com/cloudfoundry/r-buildpack.git",
                                   "Ruby": "https://github.com/cloudfoundry/ruby-buildpack.git",
                                   "Staticfile": "https://github.com/cloudfoundry/staticfile-buildpack.git"}

        if deployed.buildPack == "Custom":
            return deployed.customBuildPackURL

        return build_pack_repo_mapping[deployed.buildPack]

    @staticmethod
    def contains_bound_service(bound_service_list, bound_service):
        for bound_service_old in bound_service_list:
            if AppHelper.get_service_instance_name(
                    bound_service_old) == AppHelper.get_service_instance_name(
                bound_service) and not AppHelper.has_parameters_changed(bound_service.parameters,
                                                                                bound_service_old.parameters):
                return True

    @staticmethod
    def has_parameters_changed(old_parameters, new_parameters):
        if bool(old_parameters) != bool(new_parameters):
            return True
        if len(old_parameters) != len(new_parameters):
            return True
        if bool(old_parameters):
            for key, value in old_parameters.iteritems():
                if key not in new_parameters:
                    return True
                elif value != new_parameters[key]:
                    return True
