package com.xebialabs.deployit.plugin.codepipeline.processor;

import com.amazonaws.services.codepipeline.model.Job;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.codepipeline.actions.*;
import com.xebialabs.deployit.plugin.codepipeline.exceptions.CodePipelinePluginExecutionException;
import com.xebialabs.deployit.plugin.codepipeline.helpers.*;
import com.xebialabs.deployit.plugin.codepipeline.services.CodePipelineJobService;
import com.xebialabs.deployit.plugin.codepipeline.services.XLDeployService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JobProcessor {

    private static final Logger LOGGER = LoggerFactory.getLogger(JobProcessor.class);

    private final CodePipelineJobService jobService;
    private final XLDeployService xlDeployService;

    public JobProcessor(CodePipelineJobService jobService, XLDeployService xlDeployService) {
        this.jobService = jobService;
        this.xlDeployService = xlDeployService;
    }

    public StepExitCode processJob(Job job) {
        try {
            LOGGER.info("Processing job with id {}", job.getId());
            String environmentId = jobService.getJobData(job, Constants.ENVIRONMENT_ID);
            String deploymentPackageName = jobService.getJobData(job, Constants.DEPLOYMENT_PACKAGE_NAME);
            String deploymentPackageVersion = jobService.getJobData(job, Constants.DEPLOYMENT_PACKAGE_VERSION);

            jobService.acknowledgeJob(job);
            validate(job, deploymentPackageName, deploymentPackageVersion);
            String deploymentPackageId = getPackageId(job, deploymentPackageName, deploymentPackageVersion);
            Deployment deployment = getDeployment(environmentId, deploymentPackageId);

            validateDeployment(deployment);
            TaskExecutionState taskExecutionState = executeDeploymentTask(job, deployment);
            sendJobStatusToCodePipeline(job, taskExecutionState);
            return StepExitCode.SUCCESS;
        } catch (Exception e) {
            LOGGER.error(e.getMessage(), e);
            jobService.sendJobFailureStatus(job, e.getMessage());
            return StepExitCode.FAIL;
        }
    }

    private void sendJobStatusToCodePipeline(Job job, TaskExecutionState taskExecutionState) {
        if (taskExecutionState == TaskExecutionState.EXECUTED) {
            jobService.sendJobSuccessStatus(job, "Finished");
        } else {
            jobService.sendJobFailureStatus(job, "Deployment task failed with state " + taskExecutionState.name());
        }
    }

    private TaskExecutionState executeDeploymentTask(Job job, Deployment deployment) {
        String taskId = BeansFactory.getPrepareAction(jobService, xlDeployService, job, deployment).perform();
        TaskExecutionState state = BeansFactory.getExecuteTaskAction(xlDeployService, taskId).perform();
        LOGGER.info("Archiving task with id {}", taskId);
        xlDeployService.archiveTask(taskId);
        return state;
    }

    private void validateDeployment(Deployment deployment) {
        BeansFactory.getValidateDeploymentAction(xlDeployService, deployment).perform();
    }

    private Deployment getDeployment(String environmentId, String deploymentPackageId) {
        Action<Deployment> deploymentAction = getDeploymentAction(deploymentPackageId, environmentId);
        return deploymentAction.perform();
    }

    private Action getDeploymentAction(final String deploymentPackageId, final String environmentId) {
        return xlDeployService.isInitialDeployment(deploymentPackageId, environmentId) ?
                BeansFactory.getPrepareAction(xlDeployService, deploymentPackageId, environmentId) :
                BeansFactory.getUpdateAction(xlDeployService, deploymentPackageId, environmentId);
    }

    private String getPackageId(Job job, String deploymentPackageName,
                                String deploymentPackageVersion) {
        if (jobService.hasArtifact(job)) {
            ConfigurationItem ci = importConfigurationItem(job, deploymentPackageName, deploymentPackageVersion);
            return ci.getId();
        }
        return ConfigurationItemUtils.getId(deploymentPackageName, deploymentPackageVersion);
    }

    private ConfigurationItem importConfigurationItem(Job job, String deploymentPackageName, String deploymentPackageVersion) {
        return BeansFactory.getImportAction(xlDeployService, jobService,
                job, deploymentPackageName, deploymentPackageVersion).perform();
    }

    private void validate(Job job, String deploymentPackageName,
                          String deploymentPackageVersion) {
        if (!jobService.hasArtifact(job))
            validateExistingPackage(deploymentPackageName, deploymentPackageVersion);
    }

    private void validateExistingPackage(String deploymentPackageName,
                                              String deploymentPackageVersion) {
        if(deploymentPackageName == null || deploymentPackageVersion == null || deploymentPackageName.isEmpty() ||
         deploymentPackageVersion.isEmpty())
            throw new CodePipelinePluginExecutionException(
                    String.format("Deployment Package name and version should be specified", deploymentPackageVersion));
    }

}