#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from xld.kubernetes.commons.common_utils import CommonUtils
from xld.kubernetes.extensions_api_client import KubernetesExtensionsClient
from xld.kubernetes.ingress.helper import IngressHelper
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory


class IngressService(object):
    def __init__(self):
        self.__kind = CommonUtils.get_kind(self)

    def create(self, deployed):
        with self.__get_client(deployed) as __client:
            ingress_request = IngressHelper.read_ingress(deployed)
            print("Creating Ingress '{0}' on {1} ".
                  format(IngressHelper.get_name(deployed), IngressService.__get_container_name(deployed)))
            CommonUtils.print_request_dump(deployed.container.container, ingress_request, self.__kind,
                                           __client.get_api_version())
            ingress_response = __client.create_ingress(namespace=IngressService.__get_container_name(deployed),
                                                       ingress_body=ingress_request)
        CommonUtils.print_response_dump(deployed.container.container, ingress_response)
        return ingress_response

    def modify(self, deployed):
        with self.__get_client(deployed) as __client:
            ingress_request = IngressHelper.read_ingress(deployed)
            print("Updating Ingress '{0}' on {1} ".
                  format(IngressHelper.get_name(deployed), IngressService.__get_container_name(deployed)))
            CommonUtils.print_request_dump(deployed.container.container, ingress_request, self.__kind,
                                           __client.get_api_version())
            ingress_response = __client.replace_ingress(name=IngressHelper.get_name(deployed),
                                                        namespace=IngressService.__get_container_name(deployed),
                                                        ingress_body=ingress_request)
        CommonUtils.print_response_dump(deployed.container.container, ingress_response)
        return ingress_response

    def delete(self, previous_deployed):
        with self.__get_client(previous_deployed) as __client:
            print("Deleting Ingress '{0}' on {1}".
                  format(IngressHelper.get_name(previous_deployed), IngressService.__get_container_name(previous_deployed)))
            __client.delete_ingress(name=IngressHelper.get_name(previous_deployed),
                                    namespace=IngressService.__get_container_name(previous_deployed))

    def should_wait_for_ingress_creation_updation(self, deployed, previousDeployed):
        with self.__get_client(deployed) as __client:
            if not __client.is_ingress_existing(IngressHelper.get_name(deployed),
                                                                   IngressService.__get_container_name(deployed)):
                print("Waiting for ingress {0} to be {}.".format(IngressHelper.get_name(deployed),
                                                                 'modified' if previousDeployed else 'created'))
                return True
            return False

    def should_wait_for_ingress_deletion(self, previous_deployed):
        with self.__get_client(previous_deployed) as __client:
            if __client.is_ingress_existing(IngressHelper.get_name(previous_deployed),
                                                                        IngressService.__get_container_name(
                                                                            previous_deployed)):
                print("Waiting for ingress {0} to be deleted.".format(IngressHelper.get_name(previous_deployed)))
                return True
            return False

    def __get_client(self, deployed):
        return KubernetesExtensionsClient(deployed.container.container)

    @staticmethod
    def __get_container_name(deployed):
        return ContainerHelperFactory(deployed.container).create().get_container_name(deployed.container)
