#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from xld.kubernetes.commons.common_utils import CommonUtils
from xld.kubernetes.core_api_client import KubernetesCoreClient
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory
from xld.kubernetes.persistent_volume_claim.helper import PVCHelper
from kubernetes import client as kubernetes_client
from kubernetes.client.rest import ApiException


class PersistentVolumeClaimService(object):

    def __init__(self, deployed):
        self.__deployed = deployed
        self.__pvc_helper = PVCHelper(deployed)
        self.__container_helper = ContainerHelperFactory(deployed.container).create()
        self.__client = KubernetesCoreClient(deployed.container.container)
        self.__kind = CommonUtils.get_kind(self)

    def create(self):
        with self.__client as client:
            pvc_request = self.__pvc_helper.read_pvc()
            print("Creating PersistentVolumeClaim '{0}' in {1} '{2}' ".
                  format(self.__pvc_helper.get_name(self.__deployed), self.__container_helper.get_container_label(),
                         self.__get_container_name()))
            CommonUtils.print_request_dump(self.__deployed.container.container, pvc_request, self.__kind,
                                           client.get_api_version())
            pvc_response = client.core_api.create_namespaced_persistent_volume_claim(
                namespace=self.__get_container_name(), body=pvc_request)
        CommonUtils.print_response_dump(self.__deployed.container.container, pvc_response)
        return pvc_response

    def modify(self):
        with self.__client as client:
            pvc_request = self.__pvc_helper.read_pvc()
            print("Updating PersistentVolumeClaim '{0}' in {1} '{2}' ".
                  format(self.__pvc_helper.get_name(self.__deployed), self.__container_helper.get_container_label(),
                         self.__get_container_name()))
            CommonUtils.print_request_dump(self.__deployed.container.container, pvc_request,self.__kind,
                                           client.get_api_version())
            pvc_response = client.core_api.replace_namespaced_persistent_volume_claim(
                name=self.__pvc_helper.get_name(self.__deployed),
                namespace=self.__get_container_name(),
                body=pvc_request)
        CommonUtils.print_response_dump(self.__deployed.container.container, pvc_response)
        return pvc_response

    def destroy(self):
        with self.__client as client:
            print("Destroying PersistentVolumeClaim '{0}' from {1} '{2}' ".
                  format(self.__pvc_helper.get_name(self.__deployed), self.__container_helper.get_container_label(),
                         self.__get_container_name()))
            client.core_api.delete_namespaced_persistent_volume_claim(
                name=self.__pvc_helper.get_name(self.__deployed),
                namespace=self.__get_container_name(),
                body=kubernetes_client.V1DeleteOptions())

    def is_existing(self):
        with self.__client as client:
            try:
                pvc = client.core_api.read_namespaced_persistent_volume_claim(
                    name=self.__pvc_helper.get_name(self.__deployed),
                    namespace=self.__get_container_name())
                return bool(pvc)
            except ApiException as e:
                if e.status == 404:
                    return False

    def is_bound(self, name):
        with self.__client as client:
            pvc = client.core_api.read_namespaced_persistent_volume_claim(
                name=name, namespace=self.__get_container_name())
            is_claim_bound = bool(pvc.status.phase == 'Bound')
            print("PersistentVolumeClaim is in {} phase".format(pvc.status.phase))
        return is_claim_bound

    def __get_container_name(self):
        return self.__container_helper.get_container_name(self.__deployed.container)
