#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from xld.kubernetes.apps_api_client import KubernetesAppsClient
from xld.kubernetes.commons.common_utils import CommonUtils
from xld.kubernetes.deployment.deployment_helper import DeploymentHelper
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory


class DeploymentService(object):

    def __init__(self):
        self.__deployment_helper = DeploymentHelper()
        self.__kind = CommonUtils.get_kind(self)

    def create_deployment(self, deployed):
        with self.__create_client(deployed.container.container) as client:
            request_deployment = self.__deployment_helper.read_deployment(deployed_deployment=deployed)
            container_helper = self.__create_container_helper(deployed.container)
            print("Creating Deployment '{0}' in {1} '{2}' ".format(request_deployment['metadata']['name'],
                                                                   container_helper.get_container_label(),
                                                                   container_helper.get_container_name(
                                                                       deployed.container)))
            CommonUtils.print_request_dump(deployed.container.container, request_deployment, self.__kind,
                                           client.get_api_version())
            response_deployment = client.create_deployment(request_deployment,
                                                           container_helper.get_container_name(deployed.container))
        CommonUtils.print_response_dump(deployed.container.container, response_deployment)
        return response_deployment

    def update_deployment(self, deployed, previousDeployed):
        with self.__create_client(deployed.container.container) as client:
            request_deployment = self.__deployment_helper.read_deployment(deployed_deployment=deployed)
            if bool(deployed.restartOnConfigMapChanges):
                self.__deployment_helper.update_modifytime_label(request_deployment)
            container_helper = self.__create_container_helper(deployed.container)
            print("Updating Deployment '{0}' in {1} '{2}' ".format(
                self.__deployment_helper.get_deployment_name(previousDeployed),
                container_helper.get_container_label(),
                container_helper.get_container_name(deployed.container)))
            CommonUtils.print_request_dump(deployed.container.container, request_deployment, self.__kind,
                                           client.get_api_version())

            response_deployment = client.replace_deployment(self.__deployment_helper.get_deployment_name(previousDeployed),
                                                            request_deployment, container_helper.get_container_name(
                                                                deployed.container))
        CommonUtils.print_response_dump(deployed.container.container, response_deployment)
        return response_deployment

    def restart_deployment(self, deployed, previousDeployed):
        with self.__create_client(deployed.container.container) as client:
            request_deployment = self.__deployment_helper.read_deployment(deployed_deployment=deployed)
            self.__deployment_helper.update_modifytime_label(request_deployment)
            container_helper = self.__create_container_helper(deployed.container)
            print("Restart Deployment '{0}' in {1} '{2}' ".format(
                self.__deployment_helper.get_deployment_name(previousDeployed),
                container_helper.get_container_label(),
                container_helper.get_container_name(deployed.container)))
            response_deployment = client.replace_deployment(self.__deployment_helper.get_deployment_name(previousDeployed),
                                                            request_deployment,
                                                            container_helper.get_container_name(
                                                                deployed.container))
        return response_deployment

    def destroy_deployment(self, previousDeployed):
        with self.__create_client(previousDeployed.container.container) as client:
            container_helper = self.__create_container_helper(previousDeployed.container)
            print "Destroying Deployment '{0}' from {1} '{2}' ".format(
                self.__deployment_helper.get_deployment_name(previousDeployed),
                container_helper.get_container_label(),
                container_helper.get_container_name(previousDeployed.container))
            client.remove_deployment(self.__deployment_helper.get_deployment_name(previousDeployed),
                                     container_helper.get_container_name(
                                         previousDeployed.container))

    def should_wait_for_deployment_ready(self, deployed, deployment_name, minimum_pod_count):
        with self.__create_client(deployed.container.container) as client:
            response_deployment = client.read_deployment(deployment_name,
                                                         self.__create_container_helper(
                                                             deployed.container).get_container_name(deployed.container))
        return self.__deployment_helper.verify_deployment_ready_on_create(response_deployment, deployment_name,
                                                                          minimum_pod_count)

    def should_wait_for_deployment_ready_on_modify(self, deployed, deployment_name):
        with self.__create_client(deployed.container.container) as client:
            response_deployment = client.read_deployment(deployment_name,
                                                         self.__create_container_helper(
                                                             deployed.container).get_container_name(deployed.container))
        return self.__deployment_helper.verify_deployment_ready_on_modify(response_deployment, deployment_name)

    def should_wait_for_deployment_deletion(self, previousDeployed):
        with self.__create_client(previousDeployed.container.container) as client:
            if client.is_deployment_existing(self.__deployment_helper.get_deployment_name(previousDeployed),
                                             self.__create_container_helper(previousDeployed.container).get_container_name(
                                                 previousDeployed.container)):
                print "Waiting for Deployment {0} to be destroyed.".format(
                    self.__deployment_helper.get_deployment_name(previousDeployed))
                return True
            return False

    @staticmethod
    def __create_client(container):
        return KubernetesAppsClient(container)

    @staticmethod
    def __create_container_helper(container):
        return ContainerHelperFactory(container).create()
