#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from kubernetes import client
from kubernetes.client.rest import ApiException
from xld.kubernetes import KubernetesBaseClient
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory


class KubernetesCoreClient(KubernetesBaseClient):
    def __init__(self, container):
        super(KubernetesCoreClient, self).__init__(container)
        self.core_api = self.get_core_api()

    def get_core_api(self):
        version = self.get_api_version()
        print("[Using API version: {}]".format(version))
        return self.get_version_client(version)

    def get_api_version(self):
        return "v1"

    def create_pod(self, v1Pod, namespace):
        return self.core_api.create_namespaced_pod(namespace=namespace, body=v1Pod)

    def describe_pod(self, name, namespace, show_container_logs, bytes_container_logs):
        ret = self.core_api.list_namespaced_pod(namespace=namespace, pretty='true', watch=False)
        pods = []
        for it in ret.items:
            podname = it.metadata.name
            # filter applicable pods only
            if len(name) > 57:
                print("*********************************************************************")
                print("Results may be inaccurate if the deployment name is greater than 57.")
                print("*********************************************************************")
                name = name[0:57]
            if podname.startswith("{0}".format(name)):
                pod = {
                    'name': podname,
                    'data': it,
                    'logs': [],
                    'events': {},
                }
                # Get container logs
                if show_container_logs:
                    for cn in it.spec.containers:
                        try:
                            log = self.core_api.read_namespaced_pod_log(name=podname, pretty='true',
                                                                              container=cn.name, namespace=namespace, limit_bytes=bytes_container_logs)
                            if log:
                                pod['logs'].append(log)
                        except ApiException as e:
                            print(
                                "Unable to fetch logs for container {0} in pod {1} with namespace {2}".format(cn.name, name,
                                                                                                              namespace))
                            print("Status {0}: {1}".format(e.status, e.reason))

                # Get events for the pod
                try:
                    events = self.core_api.list_namespaced_event(
                        pretty='true',
                        namespace=namespace,
                        field_selector="involvedObject.uid={0},involvedObject.name={1},involvedObject.namespace={2}".format(
                            it.metadata.uid, podname, namespace))
                    if events:
                        pod['events'] = events
                except ApiException as e:
                    print("Unable to fetch events for pod {0} with namespace {1}".format(name, namespace))
                    print("Status {0}: {1}".format(e.status, e.reason))

                pods.append(pod)

        return pods

    def list_events(self, uid, namespace):
        return self.core_api.list_namespaced_event(
            pretty='true',
            namespace=namespace,
            field_selector="involvedObject.uid={0}".format(uid))

    def is_pod_status_ready(self, name, container):
        container_helper = ContainerHelperFactory(container).create()
        pod = self.core_api.read_namespaced_pod(name=name,
                                                      namespace=container_helper.get_container_name(container))
        is_ready = bool([condition for condition in pod.status.conditions if
                         condition.type == 'Ready' and condition.status == 'True'])
        print("Pod {0} on {1} {2} is{3} ready.".format(pod.metadata.name, container_helper.get_container_label(),
                                                       container_helper.get_container_name(container),
                                                       '' if is_ready else ' not'))
        return is_ready

    def detect_crashback_loop_off_error(self, name, container):
        container_helper = ContainerHelperFactory(container).create()
        pod = self.core_api.read_namespaced_pod(name=name,
                                                      namespace=container_helper.get_container_name(container))
        message = [status.state.waiting.message for status in pod.status.container_statuses
                   if hasattr(status.state.waiting, 'reason') and status.state.waiting.reason == 'CrashLoopBackOff']
        return message

    def is_pod_existing(self, name, namespace):
        try:
            pod = self.core_api.read_namespaced_pod(name=name, namespace=namespace)
            if pod:
                return True
        except ApiException as e:
            if e.status == 404:
                return False

    def is_namespace_existing(self, namespace):
        try:
            namespace = self.core_api.read_namespace(name=namespace)
            if namespace:
                return True
        except ApiException as e:
            if e.status == 404:
                return False

    def remove_pod(self, name, namespace):
        return self.core_api.delete_namespaced_pod(name=name, body={}, namespace=namespace)

    def replace_pod(self, name, v1Pod, namespace):
        return self.core_api.replace_namespaced_pod(namespace=namespace, body=v1Pod, name=name)

    def create_namespace(self, v1Namespace):
        return self.core_api.create_namespace(v1Namespace)

    def remove_namespace(self, v1Namespace):
        return self.core_api.delete_namespace(v1Namespace, body=client.V1DeleteOptions(), grace_period_seconds=0)

    def create_service(self, v1Service, namespace):
        return self.core_api.create_namespaced_service(namespace=namespace, body=v1Service)

    def is_service_existing(self, name, namespace):
        try:
            service = self.core_api.read_namespaced_service(name=name, namespace=namespace)
            if service:
                return True
        except ApiException as e:
            if e.status == 404:
                return False

    def read_namespaced_service(self, name, namespace):
        try:
            service = self.core_api.read_namespaced_service(name=name, namespace=namespace)
            if service:
                return service
        except ApiException as e:
            if e.status == 404:
                return False

    def remove_service(self, name, namespace):
        return self.core_api.delete_namespaced_service(name=name, namespace=namespace,
                                                             body=client.V1DeleteOptions())

    def replace_service(self, name, v1Service, namespace):
        current_service = self.read_namespaced_service(name, namespace)
        v1Service["metadata"]["resourceVersion"] = current_service.metadata.resource_version
        return self.core_api.replace_namespaced_service(namespace=namespace, body=v1Service, name=name)
