#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from xld.kubernetes.factories.handler_factory import ApiClientFactory
from kubernetes import client

from kubernetes.client.models.v1_api_group import V1APIGroup


def fix_api_group_null_value():
    """
     Monkey patch done to avoid issue while retrieving api version as per
     https://github.com/kubernetes-client/python/issues/418#issuecomment-353894663

     """
    _server_address_by_client_cid_rs = V1APIGroup.server_address_by_client_cid_rs

    @_server_address_by_client_cid_rs.setter
    def _server_address_by_client_cid_rs(self, server_address_by_client_cid_rs):
        self._server_address_by_client_cid_rs = server_address_by_client_cid_rs

    V1APIGroup.server_address_by_client_cid_rs = _server_address_by_client_cid_rs


fix_api_group_null_value()


def get_client_for_api_version(api_client, api_version):

    api_version_clients = {"admissionregistration.k8s.io/v1beta1": client.AdmissionregistrationV1beta1Api,
                            "apiextensions.k8s.io/v1beta1": client.ApiextensionsV1beta1Api,
                            "apiregistration.k8s.io/v1beta1": client.ApiregistrationV1beta1Api,
                            "apps/v1": client.AppsV1Api,
                            "apps/v1beta1": client.AppsV1beta1Api,
                            "apps/v1beta2": client.AppsV1beta2Api,
                            "authentication.k8s.io/v1": client.AuthenticationV1Api,
                            "authentication.k8s.io/v1beta1": client.AuthenticationV1beta1Api,
                            "authorization.k8s.io/v1": client.AuthorizationV1Api,
                            "authorization.k8s.io/v1beta1": client.AuthorizationV1beta1Api,
                            "autoscaling/v1": client.AutoscalingV1Api,
                            "autoscaling/v2beta1": client.AutoscalingV2beta1Api,
                            "batch/v1": client.BatchV1Api,
                            "batch/v1beta1": client.BatchV1beta1Api,
                            "batch/v2alpha1": client.BatchV2alpha1Api,
                            "certificates.k8s.io/v1beta1": client.CertificatesV1beta1Api,
                            "events.k8s.io/v1beta1": client.EventsV1beta1Api,
                            "extensions/v1beta1": client.ExtensionsV1beta1Api,
                            "networking.k8s.io/v1": client.NetworkingV1Api,
                            "policy/v1beta1": client.PolicyV1beta1Api,
                            "rbac.authorization.k8s.io/v1": client.RbacAuthorizationV1Api,
                            "rbac.authorization.k8s.io/v1alpha1": client.RbacAuthorizationV1alpha1Api,
                            "rbac.authorization.k8s.io/v1beta1": client.RbacAuthorizationV1beta1Api,
                            "scheduling.k8s.io/v1alpha1": client.SchedulingV1alpha1Api,
                            "settings.k8s.io/v1alpha1": client.SettingsV1alpha1Api,
                            "storage.k8s.io/v1": client.StorageV1Api,
                            "storage.k8s.io/v1alpha1": client.StorageV1alpha1Api,
                            "storage.k8s.io/v1beta1": client.StorageV1beta1Api,
                            "v1": client.CoreV1Api}
    return api_version_clients[api_version](api_client)


class KubernetesBaseClient(object):
    def __init__(self, container, token=None):
        self.__container = container
        self.internal_api_client = ApiClientFactory(container, token).create()

    def get_code_version(self):
        api_instance = client.VersionApi(self.internal_api_client)
        return api_instance.get_code()

    def get_preferred_version(self, api_name, supported_methods=None):
        apis_api_instance = client.ApisApi(self.internal_api_client)
        api_response = apis_api_instance.get_api_versions()
        groups = {group.name: group for group in api_response.groups}
        preferred_version = groups[api_name].preferred_version.group_version
        if supported_methods and not self.__supports_methods(preferred_version, supported_methods):
            for version in groups[api_name].versions:
                if self.__supports_methods(version.group_version, supported_methods):
                    return version.group_version
            return None
        else:
            return preferred_version

    def __supports_methods(self, version, supported_methods):
        version_client = self.get_version_client(version)
        return not bool(None in [getattr(version_client, method, None) for method in supported_methods])

    def get_version_client(self, version):
        return get_client_for_api_version(self.internal_api_client, version)

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.internal_api_client:
            self.internal_api_client.rest_client.pool_manager.__exit__(exc_type, exc_val, exc_tb)

