#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from xld.kubernetes.apps_api_client import KubernetesAppsClient
from xld.kubernetes.commons.common_utils import CommonUtils
from xld.kubernetes.statefulset.statefulset_helper import StatefulSetHelper
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory


class StatefulSetService(object):

    def __init__(self):
        self.__statefulset_helper = StatefulSetHelper()
        self.__kind = CommonUtils.get_kind(self)

    def create_statefulset(self, deployed):
        with self.__create_client(deployed.container.container, deployed.container.token) as client:
            request_statefulset = self.__statefulset_helper.read_statefulset(deployed_statefulset=deployed)
            container_helper = self.__create_container_helper(deployed.container)
            print("Creating StatefulSet '{0}' in {1} '{2}' ".format(request_statefulset['metadata']['name'],
                                                                   container_helper.get_container_label(),
                                                                   container_helper.get_container_name(
                                                                       deployed.container)))
            CommonUtils.print_request_dump(deployed.container.container, request_statefulset, self.__kind,
                                           client.get_api_version())
            response_statefulset = client.create_statefulset(request_statefulset,
                                                             container_helper.get_container_name(
                                                               deployed.container))
        CommonUtils.print_response_dump(deployed.container.container, response_statefulset)
        return response_statefulset

    def update_statefulset(self, deployed, previousDeployed):
        with self.__create_client(deployed.container.container, deployed.container.token) as client:
            request_statefulset = self.__statefulset_helper.read_statefulset(deployed_statefulset=deployed)
            if bool(deployed.restartOnConfigMapChanges):
                self.__statefulset_helper.update_modifytime_label(request_statefulset)
            container_helper = self.__create_container_helper(deployed.container)
            print("Updating StatefulSet '{0}' in {1} '{2}' ".format(
                self.__statefulset_helper.get_statefulset_name(previousDeployed),
                container_helper.get_container_label(),
                container_helper.get_container_name(deployed.container)))
            CommonUtils.print_request_dump(deployed.container.container, request_statefulset, self.__kind,
                                           client.get_api_version())

            response_statefulset = client.replace_statefulset(self.__statefulset_helper.get_statefulset_name(previousDeployed),
                                                              request_statefulset, container_helper.get_container_name(
                                                                deployed.container))
        CommonUtils.print_response_dump(deployed.container.container, response_statefulset)
        return response_statefulset

    def restart_statefulset(self, deployed, previousDeployed):
        with self.__create_client(deployed.container.container, deployed.container.token) as client:
            request_statefulset = self.__statefulset_helper.read_statefulset(deployed_statefulset=deployed)
            self.__statefulset_helper.update_modifytime_label(request_statefulset)
            container_helper = self.__create_container_helper(deployed.container)
            print("Restart StatefulSet '{0}' in {1} '{2}' ".format(
                self.__statefulset_helper.get_statefulset_name(previousDeployed),
                container_helper.get_container_label(),
                container_helper.get_container_name(deployed.container)))
            response_statefulset = client.replace_statefulset(self.__statefulset_helper.get_statefulset_name(previousDeployed),
                                                               request_statefulset,
                                                               container_helper.get_container_name(
                                                                deployed.container))
        return response_statefulset

    def destroy_statefulset(self, previousDeployed):
        with self.__create_client(previousDeployed.container.container, previousDeployed.container.token) as client:
            container_helper = self.__create_container_helper(previousDeployed.container)
            print "Destroying StatefulSet '{0}' from {1} '{2}' ".format(
                self.__statefulset_helper.get_statefulset_name(previousDeployed),
                container_helper.get_container_label(),
                container_helper.get_container_name(previousDeployed.container))
            client.remove_statefulset(self.__statefulset_helper.get_statefulset_name(previousDeployed),
                                      container_helper.get_container_name(
                                         previousDeployed.container))

    def should_wait_for_statefulset_ready(self, deployed, statefulset_name, minimum_pod_count):
        with self.__create_client(deployed.container.container, deployed.container.token) as client:
            response_statefulset = client.read_statefulset(statefulset_name,
                                                           self.__create_container_helper(
                                                             deployed.container).get_container_name(deployed.container))
        return self.__statefulset_helper.verify_stateful_set_ready_on_create(response_statefulset, statefulset_name,
                                                                             minimum_pod_count)

    def should_wait_for_statefulset_ready_on_modify(self, deployed, statefulset_name, minimum_pod_count):
        with self.__create_client(deployed.container.container, deployed.container.token) as client:
            response_statefulset = client.read_statefulset(statefulset_name,
                                                         self.__create_container_helper(
                                                             deployed.container).get_container_name(deployed.container))
        return self.__statefulset_helper.verify_stateful_set_ready_on_create(response_statefulset, statefulset_name,
                                                                             minimum_pod_count)

    def should_wait_for_statefulset_deletion(self, previousDeployed):
        with self.__create_client(previousDeployed.container.container, previousDeployed.container.token) as client:
            if client.is_statefulset_existing(self.__statefulset_helper.get_statefulset_name(previousDeployed),
                                              self.__create_container_helper(previousDeployed.container).get_container_name(
                                                 previousDeployed.container)):
                print "Waiting for StatefulSet {0} to be destroyed.".format(
                    self.__statefulset_helper.get_statefulset_name(previousDeployed))
                return True
            return False

    @staticmethod
    def __create_client(container, token=None):
        return KubernetesAppsClient(container, token)

    @staticmethod
    def __create_container_helper(container):
        return ContainerHelperFactory(container).create()
