from kubernetes import client
from kubernetes.client.api_client import ApiClient
from xld.kubernetes.pod.pod_helper import PodHelper


class DeploymentHelper(object):

    def __init__(self):
        self.__pod_helper = PodHelper()

    def read_deployment(self, deployed_deployment):
        deployment = client.V1beta1Deployment()
        deployment.metadata = client.V1ObjectMeta(name=self.get_deployment_name(deployed_deployment))

        spec = client.V1beta1DeploymentSpec()
        spec.min_ready_seconds = deployed_deployment.minimumReadySeconds
        spec.paused = deployed_deployment.paused
        spec.progress_deadline_seconds = deployed_deployment.progressDeadlineSeconds
        spec.replicas = deployed_deployment.replicasCount
        spec.revision_history_limit = deployed_deployment.revisionHistoryLimit

        if deployed_deployment.matchExpressions or deployed_deployment.matchLabels:
            spec.selector = self.__read_selector(deployed_deployment)

        if deployed_deployment.strategyType:
            spec.strategy = self.__read_deployment_strategy(deployed_deployment)


        spec = ApiClient().sanitize_for_serialization(spec)
        spec['template'] = self.__pod_helper.read_pod(deployed_pod=deployed_deployment, pod=client.V1PodTemplateSpec())
        deployment = ApiClient().sanitize_for_serialization(deployment)
        deployment['spec'] = spec
        return deployment

    def get_deployment_name(self, deployed):
        return deployed.deploymentName if deployed.deploymentName else deployed.name

    @staticmethod
    def __read_match_expression(deployed_match_expression):
        match_expression = client.UnversionedLabelSelectorRequirement()
        match_expression.key = deployed_match_expression.key
        match_expression.operator = deployed_match_expression.operator
        if deployed_match_expression.matchValues:
            match_expression.values = deployed_match_expression.matchValues
        return match_expression

    @staticmethod
    def __read_selector(deployed_deployment):
        selector = client.UnversionedLabelSelector()
        if deployed_deployment.matchExpressions:
            selector.match_expressions = []
            for deployed_match_expression in deployed_deployment.matchExpressions:
                selector.match_expressions.append(DeploymentHelper.__read_match_expression(deployed_match_expression))
        if deployed_deployment.matchLabels:
            selector.match_labels = deployed_deployment.matchLabels

        return selector

    @staticmethod
    def __read_deployment_strategy(deployed_deployment):
        strategy = client.V1beta1DeploymentStrategy()
        if deployed_deployment.maximumSurge or deployed_deployment.maximumnUnAvailable:
            strategy.rolling_update = client.V1beta1RollingUpdateDeployment()
            strategy.rolling_update.max_surge = deployed_deployment.maximumSurge
            strategy.rolling_update.max_unavailable = deployed_deployment.maximumnUnAvailable
        strategy.type = deployed_deployment.strategyType
        return strategy
