#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from kubernetes import client
from kubernetes.client.api_client import ApiClient
from xld.kubernetes.pod.pod_helper import PodHelper
import time


class DeploymentHelper(object):
    def __init__(self):
        self.__pod_helper = PodHelper()

    def read_deployment(self, deployed_deployment):
        deployment = client.V1beta2Deployment()
        deployment.metadata = client.V1ObjectMeta(name=self.get_deployment_name(deployed_deployment))

        template = self.__pod_helper.read_pod(deployed_pod=deployed_deployment, pod=client.V1PodTemplateSpec())
        template['metadata']['name'] = deployment.metadata.name
        spec = client.V1beta2DeploymentSpec(template=template, selector=self.__read_selector(deployed_deployment))
        spec.min_ready_seconds = deployed_deployment.minimumReadySeconds
        spec.paused = deployed_deployment.paused
        spec.progress_deadline_seconds = deployed_deployment.progressDeadlineSeconds
        spec.replicas = deployed_deployment.replicasCount
        spec.revision_history_limit = deployed_deployment.revisionHistoryLimit

        if deployed_deployment.strategyType:
            spec.strategy = self.__read_deployment_strategy(deployed_deployment)

        spec = ApiClient().sanitize_for_serialization(spec)
        deployment = ApiClient().sanitize_for_serialization(deployment)
        deployment['spec'] = spec
        return deployment

    def update_modifytime_label(self, deployment):
        deployment['spec']['template']['metadata']['labels']['modifytime'] = str(int(time.time()))

    def get_deployment_name(self, deployed):
        return deployed.deploymentName if deployed.deploymentName else deployed.name

    @staticmethod
    def __read_match_expression(deployed_match_expression):
        match_expression = client.V1LabelSelectorRequirement(key=deployed_match_expression.key,
                                                             operator=deployed_match_expression.operator)
        if deployed_match_expression.matchValues:
            match_expression.values = deployed_match_expression.matchValues
        return match_expression

    @staticmethod
    def __read_selector(deployed_deployment):
        selector = client.V1LabelSelector()
        if deployed_deployment.matchExpressions:
            selector.match_expressions = []
            for deployed_match_expression in deployed_deployment.matchExpressions:
                selector.match_expressions.append(DeploymentHelper.__read_match_expression(deployed_match_expression))
        if deployed_deployment.matchLabels:
            selector.match_labels = deployed_deployment.matchLabels

        return selector

    @staticmethod
    def __read_deployment_strategy(deployed_deployment):
        strategy = client.V1beta2DeploymentStrategy()
        if deployed_deployment.maximumSurge or deployed_deployment.maximumUnAvailable:
            strategy.rolling_update = client.V1beta2RollingUpdateDeployment()
            strategy.rolling_update.max_surge = deployed_deployment.maximumSurge
            strategy.rolling_update.max_unavailable = deployed_deployment.maximumUnAvailable
        strategy.type = deployed_deployment.strategyType
        return strategy

    @staticmethod
    def validate_deployment(deployed_deployment):
        if deployed_deployment.minimumPodCount > deployed_deployment.replicasCount:
            raise RuntimeError("Minimum pod count should not be more than replicas count.")

        if not deployed_deployment.matchExpressions and not deployed_deployment.matchLabels:
            raise RuntimeError("Match Expressions or Match Labels should not be empty.")

        if not deployed_deployment.labels:
            raise RuntimeError("Labels should not be empty.")

    @staticmethod
    def verify_deployment_ready_on_create(response_deployment, deployment_name, minimum_pod_count):
        if response_deployment.status.available_replicas < minimum_pod_count:
            print("Waiting for Deployment's pod {0} to be in running state".format(deployment_name))
            return True
        return False

    @staticmethod
    def verify_deployment_ready_on_modify(response_deployment, deployment_name):
        if response_deployment.status.unavailable_replicas > 0:
            print("Waiting for Deployment's pod {0} to be in running state".format(deployment_name))
            return True
        return False
