#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

import socket
import string

setattr(socket, 'SOL_TCP', 6)

from kubernetes.config.kube_config import KubeConfigLoader
from kubernetes.client import ApiClient
from xld.kubernetes.commons.common_utils import CommonUtils
from com.xebialabs.deployit.plugin.kubernetes import LoaderUtil
from java.nio.file import Files, Paths, StandardCopyOption
import tempfile
import base64
import os
from kubernetes.client.configuration import Configuration

class KubernetesClientFactory(object):
    def __init__(self, container):
        self.__container = container
        self.__context = "kb_context"
        self.__cluster = "kb_cluster"
        self.__user = "kb_user"

    def create(self):
        cluster = self._create_cluster(server_url=self.__container.apiServerURL, skip_tls=self.__container.skipTLS)
        user = self._create_user(token=self.__container.token, username=self.__container.username,
                                 password=self.__container.password)
        context = self._create_context()
        config_params = self._get_config_params(cluster=cluster, context=context, user=user)
        return ApiClient(configuration=self.__create_config(config_params))

    def _create_cluster(self, server_url=None, skip_tls=True):
        cluster = {"server": server_url,
                   'insecure-skip-tls-verify': "true" if skip_tls else None,
                   "certificate-authority-data": base64.b64encode(
                       self._correct(self.__container.caCert)) if self.__container.caCert else None}
        return CommonUtils.remove_none_keys(cluster)

    def _create_context(self):
        return {"cluster": self.__cluster, "user": self.__user}

    def _create_user(self, token=None, username=None, password=None):
        user = {
            "client-certificate-data": base64.b64encode(self._correct(self.__container.tlsCert)) if self.__container.tlsCert else None,
            "client-key-data": base64.b64encode(
                self._correct(self.__container.tlsPrivateKey)) if self.__container.tlsPrivateKey else None,
            "token": token, "username": username,
            "password": password}
        return CommonUtils.remove_none_keys(user)

    def _get_config_params(self, context, cluster, user):
        return {"current-context": self.__context,
                "contexts": [{"name": self.__context, "context": context}],
                "clusters": [{"name": self.__cluster, "cluster": cluster}],
                "users": [{"name": self.__user, "user": user}]}

    def __create_config(self, config_params):
        client_config = Configuration()
        config_loader = KubeConfigLoader(config_dict=config_params, active_context=self.__context)
        config_loader.load_and_set(client_config)
        if self.__container.skipTLS:
            self._disable_https_warnings()
            client_config.ssl_ca_cert = self._set_ca_bundle_path(path="certifi/cacert.pem")
        return client_config

    @staticmethod
    def _disable_https_warnings():
        import urllib3
        urllib3.disable_warnings()

    @staticmethod
    def _correct(cert):
        if "-----" in cert and " " in cert.split('-----')[2]:
            return KubernetesClientFactory._cert_replace_spaces(cert).strip()
        else:
            return cert.strip()


    @staticmethod
    def _cert_replace_spaces(cert):
        splits=cert.split('-----')
        splits[2]=string.replace(splits[2], ' ', '\n')
        return '-----'.join(splits)

    def _set_ca_bundle_path(self, path):
        ca_bundle_path = self.__extract_file_from_jar(path)
        os.environ['REQUESTS_CA_BUNDLE'] = ca_bundle_path
        return ca_bundle_path

    def __extract_file_from_jar(self, config_file):
        file_url = LoaderUtil.getResourceBySelfClassLoader(config_file)
        if file_url:
            tmp_file, tmp_abs_path = tempfile.mkstemp()
            tmp_file.close()
            Files.copy(file_url.openStream(), Paths.get(tmp_abs_path), StandardCopyOption.REPLACE_EXISTING)
            return tmp_abs_path
        else:
            return None
