#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from kubernetes import client
from xld.kubernetes.pod.pod_helper import PodHelper
from kubernetes.client.api_client import ApiClient
from xld.kubernetes.commons.common_utils import CommonUtils
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory


class ExposedServiceStepsHelper(object):

    def __init__(self, context, steps):
        self.__context = context
        self.__steps = steps

    @staticmethod
    def get_exposed_services(containers, labels, service_prefix):
        services = []
        if bool(containers):
            for container in containers:
                if bool(container.ports):
                    for port in container.ports:
                        if port.exposeAsService:
                            service = ExposedServiceStepsHelper.__get_exposed_service(port, labels,
                                                                                  "{}-{}-{}".format(service_prefix,
                                                                                                    PodHelper.get_container_name(
                                                                                                        container),
                                                                                                    port.name))
                            services.append(ApiClient().sanitize_for_serialization(service))
        return services

    @staticmethod
    def __get_exposed_service(container_port, labels, service_name_generated):
        service = client.V1Service()
        service.metadata = client.V1ObjectMeta(
            name=container_port.serviceName if container_port.serviceName else service_name_generated)
        spec = client.V1ServiceSpec()
        spec.type = container_port.serviceType if bool(container_port.serviceType) else 'NodePort'
        spec.selector = labels
        spec.ports = [client.V1ServicePort(name='http',
                                           port=container_port.servicePort if container_port.servicePort else (
                                               container_port.hostPort if container_port.hostPort else container_port.containerPort),
                                           target_port=CommonUtils.convert_to_int_if_possible(
                                               container_port.containerPort))]
        spec.load_balancer_ip = container_port.loadBalancerIP
        spec = ApiClient().sanitize_for_serialization(spec)
        service.spec = spec
        return service

    def add_create_steps(self, deployed, new_services, delta):
        container_helper = ContainerHelperFactory(deployed.container).create()
        for new_service in new_services:
            self.__context.addStepWithCheckpoint(self.__steps.jython(
                description="Create Service {0} on {1}".format(new_service['metadata']['name'],
                                                               container_helper.get_container_name(deployed.container)),
                script="xld/kubernetes/service/exposedservices/create_service.py",
                jython_context={"deployed": deployed, "service": new_service},
                order=67
            ), delta)

            self.__context.addStep(self.__steps.jython(
                description="Wait for Service {0} to be created".format(new_service['metadata']['name']),
                script="xld/kubernetes/service/exposedservices/wait_for_service_creation.py",
                jython_context={"deployed": deployed, "service": new_service},
                order=68
            ))

    def add_modify_steps(self, deployed, previousDeployed, old_services, new_services, delta):
        services_to_be_destroyed = [old_service for old_service in old_services if
                                    not ExposedServiceStepsHelper.__contains_service(new_services, old_service)]
        self.add_destroy_steps(previousDeployed, services_to_be_destroyed, delta)
        services_to_be_created = [new_service for new_service in new_services if
                                  not ExposedServiceStepsHelper.__contains_service(old_services, new_service)]
        self.add_create_steps(deployed, services_to_be_created, delta)

    def add_destroy_steps(self, previousDeployed, old_services, delta):
        container_helper = ContainerHelperFactory(previousDeployed.container).create()
        for old_service in old_services:
            self.__context.addStepWithCheckpoint(self.__steps.jython(
                description="Destroy Service {0} from {1}".format(old_service['metadata']['name'],
                                                                  container_helper.get_container_name(
                                                                      previousDeployed.container)),
                script="xld/kubernetes/service/exposedservices/remove_service.py",
                jython_context={"previousDeployed": previousDeployed, "service": old_service},
                order=33
            ), delta)

            self.__context.addStep(self.__steps.jython(
                description="Wait for Service {0} to be destroyed completely".format(old_service['metadata']['name']),
                script="xld/kubernetes/service/exposedservices/wait_for_service_deletion.py",
                jython_context={"previousDeployed": previousDeployed, "service": old_service},
                order=34
            ))

    @staticmethod
    def __contains_service(service_list, service):
        for item in service_list:
            if item['metadata']['name'] == service['metadata']['name'] and item['spec']['type'] == service['spec'][
                'type'] and item['spec']['selector']['app'] == service['spec']['selector']['app'] \
                and item['spec']['ports'][0]['port'] == service['spec']['ports'][0]['port'] and \
                item['spec']['ports'][0]['target_port'] == service['spec']['ports'][0]['target_port']:
                return True
        return False
