'use strict';

angular.module('xld.templates', ['xld.templates.application', 'xld.templates.common']);

angular.module('NewApplication', ['ui.router', 'xl.components', 'xld.templates', 'xl.templates', 'ui.bootstrap', 'xl.online.commons']).config(["$stateProvider", "$urlRouterProvider", function ($stateProvider, $urlRouterProvider) {
    $urlRouterProvider.otherwise('/new-application');
    $stateProvider.state('new-application', {
        url: '/new-application',
        templateUrl: 'application/applications.html',
        controller: 'ApplicationsManagerController',
        controllerAs: 'vm',
        bindToController: true
    });
}]).constant('xlDeployServicesRoot', 'deployit/').run(["$rootScope", "typeSystemService", "dipHandlers", function ($rootScope, typeSystemService, dipHandlers) {
    typeSystemService.loadTypesDescriptors();

    // Used in DIP Forms
    $rootScope.dipHandlers = dipHandlers;
}]).value('localeSupported', ['en-US']);
'use strict';

angular.module('NewApplication').controller('DeployableModalController', ["$scope", "$xlModalInstance", "modalData", "typeSystemService", "confirmDialogService", function ($scope, $xlModalInstance, modalData, typeSystemService, confirmDialogService) {
    var askConfirmation = function askConfirmation() {
        confirmDialogService.openConfirmDialog({
            message: 'Are you sure you want to leave? Any changes will be lost.',
            confirm: 'Yes', reject: 'Cancel', showIf: !$scope.formDeployableCiModel.form.$pristine
        }).then(function () {
            return $xlModalInstance.dismiss();
        });
    };

    $scope.isSourceArtifact = modalData.isSourceArtifact;
    $scope.displayMode = modalData.displayMode;
    $scope.deployableCiModel = modalData.model || { form: {} };
    $scope.formDeployableCiModel = { form: {} };
    $scope.typesDeployableCi = typeSystemService.getDeployableTypes();

    $xlModalInstance.handlers = {
        onEscModal: askConfirmation,
        onCloseModal: askConfirmation
    };

    $scope.cancel = function () {
        return askConfirmation();
    };

    $scope.isDisabledSaveButton = function () {
        var deployableModelForm = $scope.formDeployableCiModel.form;
        if (!deployableModelForm) {
            return true;
        }

        if (!$scope.isSourceArtifact || $scope.displayMode !== 'creation') {
            return !deployableModelForm.$valid;
        }
        var uploadFile = $scope.deployableCiModel.form.__upload_file__;
        var fileChosen = $scope.deployableCiModel.remoteFileUri || uploadFile && uploadFile.file;
        return !deployableModelForm.$valid || _.isUndefined(fileChosen);
    };

    $scope.create = function () {
        if ($scope.isSourceArtifact && $scope.displayMode === 'creation') {
            $scope.deployableCiModel.form.fileUri = $scope.deployableCiModel.remoteFileUri;
        }
        $xlModalInstance.close($scope.deployableCiModel);
    };
}]);
'use strict';

angular.module('NewApplication').factory('importResource', ["$http", "xlDeployServicesRoot", function ($http, xlDeployServicesRoot) {

    var importFormURL = function importFormURL(url, user, password) {
        var request = {
            url: url, user: user, password: password
        };
        return $http.post(xlDeployServicesRoot + 'package/fetch2', request);
    };

    var importPackageFromServer = function importPackageFromServer(id) {
        return $http.post(xlDeployServicesRoot + 'package/import/' + id);
    };

    var importablePackage = function importablePackage() {
        return $http.get(xlDeployServicesRoot + 'package/import');
    };

    return {
        importFormURL: importFormURL,
        importPackageFromServer: importPackageFromServer,
        importablePackage: importablePackage
    };
}]);
'use strict';

angular.module('NewApplication').controller('ImportPackageController', ["$scope", "$q", "$timeout", "$http", "xlDeployServicesRoot", "importResource", function ($scope, $q, $timeout, $http, xlDeployServicesRoot, importResource) {
    $scope.oneAtATime = true;

    var vm = this;
    var parentVm = $scope.$parent.vm;

    // Accordion
    vm.importObject = {
        fromUrl: { state: undefined, message: '', model: undefined },
        fromDisk: { state: undefined, message: '', model: undefined },
        fromServer: { state: undefined, message: '', model: undefined }
    };

    // Import From URL -------------------------------------------------------
    vm.importUrl = function () {
        vm.importObject.fromUrl.state = 'importing';
        vm.importObject.fromUrl.message = 'Importing...';

        // TODO: To be deleted
        var testDelay = vm.importObject.fromUrl.model === 'test' ? 5000 : 0;

        $timeout(function () {
            importResource.importFormURL(vm.importObject.fromUrl.model, vm.importObject.fromUrl.user, vm.importObject.fromUrl.password).then(function (response) {
                if (!!response && !!response.data && !!response.data.id) {
                    var packageId = response.data.id.replace('Applications/', '');
                    parentVm.unfoldPathToNode(packageId);
                    vm.importObject.fromUrl.model = '';
                    vm.importObject.fromUrl.user = undefined;
                    vm.importObject.fromUrl.password = undefined;
                    vm.importObject.fromUrl.state = 'success';
                    vm.importObject.fromUrl.message = 'Imported ' + packageId;
                    vm.importPackageForm.$setPristine();
                } else {
                    vm.importObject.fromUrl.state = 'error';
                    vm.importObject.fromUrl.message = 'Error: empty response';
                }
            }, function (error) {
                if (error.xlPreventDefault) {
                    vm.importObject.fromUrl.state = undefined;
                } else {
                    vm.importObject.fromUrl.state = 'error';
                    vm.importObject.fromUrl.message = error.data ? 'Error: ' + error.data : 'Unknown error';
                }
            });
        }, testDelay);
    };

    $scope.$watch('importObject.fromUrl.model', function (value) {
        if (value) {
            if (vm.importObject.fromUrl.stateOld === 'success' || vm.importObject.fromUrl.state === 'error') {
                vm.importObject.fromUrl.state = undefined;
                vm.importObject.fromUrl.message = '';
            }
            vm.importObject.fromUrl.stateOld = vm.importObject.fromUrl.state;
            $scope.$setTabPristine(); //eslint-disable-line
        }
    });

    // Import From Disk -----------------------------------------------------

    var onStartUpload = function onStartUpload() {
        vm.importObject.fromDisk.state = 'importing';
    };

    var onSuccessFileUpload = function onSuccessFileUpload(data) {
        vm.importObject.fromDisk.state = 'success';
        var packageId = data.result.id.replace('Applications/', '');
        parentVm.unfoldPathToNode(packageId);
        return 'Imported ' + packageId;
    };

    var onErrorFileUpload = function onErrorFileUpload(data) {
        vm.importObject.fromDisk.state = 'error';
        var responseText = _.get(data, ['jqXHR', 'responseText']);
        return !_.isEmpty(responseText) ? 'Error: ' + responseText : '';
    };

    vm.fileUploadOptions = {
        uploadUrl: xlDeployServicesRoot + 'package/upload/file.dar',
        fileNameField: 'fileData',
        authorization: $http.defaults.headers.common.Authorization,
        onStartUpload: onStartUpload,
        onSuccess: onSuccessFileUpload,
        onError: onErrorFileUpload
    };

    // Import From Server ----------------------------------------------------
    var serverPackages = [];
    importResource.importablePackage().then(function (response) {
        serverPackages = response.data;
    });

    vm.autocompleteImportHandlers = {
        addCandidates: function addCandidates() {
            var differed = $q.defer();
            differed.resolve(serverPackages);
            return differed.promise;
        }
    };

    vm.importPackageFromServer = function () {
        vm.importObject.fromServer.state = 'importing';
        vm.importObject.fromServer.message = 'Importing...';
        importResource.importPackageFromServer(vm.importObject.fromServer.model).then(function (response) {
            var packageId = response.data.id.replace('Applications/', '');
            parentVm.unfoldPathToNode(packageId);
            vm.importObject.fromServer.state = 'success';
            vm.importObject.fromServer.message = 'Imported ' + packageId;
        }, function (error) {
            vm.importObject.fromServer.state = 'error';
            vm.importObject.fromServer.message = error.data ? 'Error: ' + error.data : undefined;
        });
    };

    $scope.$watch('importObject.fromServer.model', function (value) {
        vm.importObject.fromServer.state = undefined;
        vm.importObject.fromServer.message = '';
        if (!_.isEmpty(value)) {
            $scope.$setTabPristine(); //eslint-disable-line
        }
    });
}]);
'use strict';

angular.module('NewApplication').factory('deploymentPackageService', ["$q", "$log", "ciService", "applicationsResource", "ciResource", "constantService", "utils", "$xlModal", "confirmDialogService", "typeSystemService", function ($q, $log, ciService, applicationsResource, ciResource, constantService, utils, $xlModal, confirmDialogService, typeSystemService) {
    function getDeploymentPackage(versionId) {
        var deploymentPackage = {};

        return ciResource.get(versionId).then(function (response) {
            var ci = response.data;

            var data = ciService.extractCiStructure(ci.id);
            deploymentPackage.existingApplicationName = { id: data.path };
            deploymentPackage.applicationVersion = data.name;
            deploymentPackage.versionPropertiesFormData = ci;

            // Load deployable
            deploymentPackage.deployableArtifacts = [];
            deploymentPackage.deployableResources = [];

            if (response.data.deployables && !_.isEmpty(response.data.deployables)) {
                return ciResource.findAllCisByAncestor(versionId).then(function (cis) {
                    var deployables = cis.data;
                    _.forEach(deployables, function (deployable) {
                        var deployableData = ciService.extractCiStructure(deployable.ref);
                        var item = {
                            lazy: true,
                            name: deployableData.name,
                            ciId: { id: deployableData.id }
                        };

                        if (typeSystemService.isSourceArtifactType(deployable.type)) {
                            item.uploaded = true;
                            deploymentPackage.deployableArtifacts.push(item);
                        } else {
                            deploymentPackage.deployableResources.push(item);
                        }
                    });
                    return deploymentPackage;
                });
            } else {
                return deploymentPackage;
            }
        });
    }

    function addCandidatesApplication(meta, params) {
        return applicationsResource.findAllApplications(params.term).then(function (response) {
            return _.map(response.data, function (ci) {
                ci.id = utils.removePrefix(ci.ref);
                return ci;
            });
        });
    }

    function saveCisApplicationAndVersion(cis) {
        var promises = [];

        _.forEach(cis, function (ci) {
            var deferred = $q.defer();
            ciResource.ciExists(ci.id).then(function (response) {
                var exists = response.data.boolean;
                if (!exists || exists && ci.type === constantService.TYPE.UDM_DEPLOYMENT_PACKAGE) {
                    return deferred.resolve(ci);
                }
                return deferred.resolve();
            });
            promises.push(deferred.promise);
        });

        return $q.all(promises).then(function (allCis) {
            return ciResource.saveAllCis(_.without(allCis, undefined));
        });
    }

    /********************************* Deployable methods *********************************/
    var deleteDeployable = function deleteDeployable(ciId, message) {
        return confirmDialogService.openConfirmDialog({ message: message, confirm: 'Yes', reject: 'Cancel' }).then(function () {
            return ciResource.deleteCi(ciId);
        });
    };

    var deleteFile = function deleteFile(ciId) {
        return deleteDeployable(ciId, 'Are you sure you want to delete the file?');
    };

    var deleteResource = function deleteResource(ciId) {
        return deleteDeployable(ciId, 'Are you sure you want to delete the resource?');
    };

    var openDeployableModal = function openDeployableModal(path, isSourceArtifact, displayMode, model) {
        function openModal() {
            var modalOptions = {
                templateUrl: 'application/tabs/deployment-package/deployable-modal/deployable-modal.html',
                windowClass: 'xl-deployable-modal',
                controller: 'DeployableModalController',
                resolve: {
                    modalData: function modalData() {
                        return {
                            isSourceArtifact: isSourceArtifact,
                            displayMode: displayMode,
                            model: model
                        };
                    }
                }
            };

            return $xlModal.open(modalOptions).result;
        }

        if (model && !!model.lazy) {
            return ciResource.get(model.ciId.id).then(function (response) {
                var ci = response.data;
                model.type = ci.type;
                model.form = ci;

                delete model.form.$token;
                delete model.form.id;
                delete model.form.type;
                delete model.lazy;
                return openModal();
            });
        } else {
            return openModal();
        }
    };

    var saveDeployableCi = function saveDeployableCi(path, deployable) {
        function update() {
            var ci = ciService.createJsonCiWithData(deployable.ciId, deployable.type, deployable.form);
            return ciResource.editCi(deployable.ciId.id, ci);
        }

        function save() {
            var uploadFile = deployable.form.__upload_file__;
            var clonedForm = angular.copy(deployable.form);
            delete clonedForm.__upload_file__;

            var ci = ciService.createJsonCiWithData(deployable.ciId, deployable.type, clonedForm);

            if (uploadFile && uploadFile.file) {
                return ciResource.saveCiMultipart(ci, uploadFile.file);
            } else {
                return ciResource.saveCi(ci);
            }
        }

        var isCreation = _.isUndefined(deployable.ciId);
        deployable.ciId = ciService.normalizeCiId(constantService.ROOT.APPLICATIONS, path, deployable.name);

        return (isCreation ? save() : update()).then(function () {
            if (deployable.progressStatus) {
                deployable.progressStatus = 'success';
            }
            return deployable;
        }, function (reason) {
            if (deployable.progressStatus) {
                $log.log('Error upload', reason);
                deployable.progressStatus = 'error';
            }
            return deployable;
        }, function (evt) {
            if (evt && evt.status === 'progress') {
                deployable.progress = evt.loaded;
                if (deployable.fileSize !== evt.total) {
                    deployable.fileSize = evt.total;
                }

                if (deployable.cancelUpload) {
                    evt.abort();
                }
            }
        });
    };

    function editCiVersion(versionCiObj, type, data) {
        return ciResource.editCi(versionCiObj.id, ciService.createJsonCiWithData(versionCiObj, type, data));
    }

    return {
        getDeploymentPackage: getDeploymentPackage,
        addCandidatesApplication: addCandidatesApplication,
        saveCisApplicationAndVersion: saveCisApplicationAndVersion,
        deleteFile: deleteFile,
        deleteResource: deleteResource,
        openDeployableModal: openDeployableModal,
        editCiVersion: editCiVersion,
        saveDeployableCi: saveDeployableCi
    };
}]);
'use strict';

angular.module('NewApplication').filter('percentage', ["$window", function ($window) {
    return function (input) {
        if ($window.isNaN(input)) {
            return '';
        }
        return Math.round(input * 100) + '%';
    };
}]);

angular.module('NewApplication').filter('humanFileSize', ["$window", function ($window) {
    var sizes = ['KB', 'MB', 'GB', 'TB'];

    return function (input) {
        if ($window.isNaN(input)) {
            return '';
        }
        if (input < 1024) {
            return Number(input) + ' Bytes';
        }

        var u = -1;
        var currentInput = input;
        do {
            currentInput = currentInput / 1024;
            ++u;
        } while (currentInput >= 1024);
        return currentInput.toFixed() + ' ' + sizes[u];
    };
}]);
'use strict';

angular.module('NewApplication').controller('DeploymentPackageController', ["$scope", "$interval", "deploymentPackageService", "ciService", "constantService", "typeSystemService", "deployableService", "utils", "applicationsService", "$uiTabs", "humanFileSizeFilter", "percentageFilter", function ($scope, $interval, deploymentPackageService, ciService, constantService, typeSystemService, deployableService, utils, applicationsService, $uiTabs, humanFileSizeFilter, percentageFilter) {
    var vm = this;
    var parentVm = $scope.$parent.vm;

    vm.versionTypeFormValidation = { form: {} };
    vm.versionPropertiesFormData = {};
    vm.showEditPackageOptions = false;

    // Pre-load data
    var tab = $uiTabs.getTab($scope.$$tabId);
    if (tab) {
        vm.mode = tab.mode;

        if (tab.versionId) {
            deploymentPackageService.getDeploymentPackage(tab.versionId).then(function (deploymentPackage) {
                vm.existingApplicationName = deploymentPackage.existingApplicationName;
                vm.applicationVersion = deploymentPackage.applicationVersion;
                vm.idVersionWithoutRoot = utils.removePrefix(tab.versionId);
                vm.versionPropertiesFormData = deploymentPackage.versionPropertiesFormData;
                vm.deployableArtifacts = deploymentPackage.deployableArtifacts;
                vm.deployableResources = deploymentPackage.deployableResources;
                vm.showDeploymentDetails = true;
            });
        }
    }

    vm.existingApplicationNameHandlers = {
        addCandidates: deploymentPackageService.addCandidatesApplication
    };

    $scope.$watch('applicationVersion', function () {
        delete vm.postError;
        delete vm.configurationError;
    });

    vm.createAndConfigure = function () {
        delete vm.postError;

        // Prepare Json
        var newPathAndAppName = utils.addLeadingSlashIfNotExists(vm.existingApplicationName.id);
        var appObj = ciService.extractCiStructure(constantService.ROOT.APPLICATIONS + newPathAndAppName);

        var cis = [];
        if (!vm.existingApplicationName.type) {
            cis = ciService.createDirectoriesJsonCis(appObj);
            var appCi = ciService.createJsonCi(appObj);
            appCi.type = constantService.TYPE.UDM_APPLICATIONS;
            cis.push(appCi);
        }

        var versionObj = ciService.normalizeCiId(constantService.ROOT.APPLICATIONS, appObj.idWithoutRoot, vm.applicationVersion);
        cis.push(ciService.createJsonCi(versionObj, constantService.TYPE.UDM_DEPLOYMENT_PACKAGE));

        // SaveJson
        deploymentPackageService.saveCisApplicationAndVersion(cis).then(function () {
            parentVm.unfoldPathToNode(versionObj.idWithoutRoot);
            vm.idVersionWithoutRoot = versionObj.idWithoutRoot;
            vm.showDeploymentDetails = true;
            vm.showEditPackageOptions = true;
        }, function (error) {
            vm.postError = error.data && !error.xlPreventDefault ? error.data : undefined;
        });
    };

    // Deployable Artifacts
    vm.addDeployableArtifact = function () {
        deploymentPackageService.openDeployableModal(vm.idVersionWithoutRoot, true, constantService.MODES.CREATION).then(function (artifact) {
            if (artifact.form.fileUri) {
                artifact.progress = 100;
                artifact.stripped = true;
            } else {
                artifact.onCancelUpload = function () {
                    artifact.cancelUpload = true;
                };
            }
            artifact.progressStatus = 'inProgress';
            artifact.uploaded = false;
            vm.deployableArtifacts.push(artifact);

            deploymentPackageService.saveDeployableCi(vm.idVersionWithoutRoot, artifact).then(function () {
                if (artifact.progressStatus === 'success') {
                    artifact.uploaded = true;
                    delete artifact.progress;
                    parentVm.unfoldPathToNode(artifact.ciId.idWithoutRoot);
                } else if (artifact.progressStatus === 'error') {
                    _.remove(vm.deployableArtifacts, ['ciId.id', artifact.ciId.id]); //TODO: verify
                }
            });
        });
    };

    vm.showProgressMessage = function (artifact) {
        var progressPercent = artifact.progress / artifact.fileSize;
        if (progressPercent < 1) {
            return 'Imported ' + humanFileSizeFilter(artifact.progress) + ' of ' + humanFileSizeFilter(artifact.fileSize) + ' ' + percentageFilter(progressPercent);
        }
        return 'Saving....';
    };

    vm.editDeployableArtifact = function (id) {
        var model = _.find(vm.deployableArtifacts, { ciId: { id: id } });
        deploymentPackageService.openDeployableModal(vm.idVersionWithoutRoot, true, constantService.MODES.EDITION, model).then(function (artifact) {
            delete artifact.form.__upload_file__;
            deployableService.editDeployable(artifact.ciId, artifact.type, artifact.form).then(function () {
                var idWithoutRoot = artifact.ciId.id.substring(artifact.ciId.id.indexOf('/'));
                parentVm.unfoldPathToNode(idWithoutRoot);
            });
        });
    };

    vm.deleteDeployableArtifact = function (ciId) {
        deploymentPackageService.deleteFile(ciId).then(function () {
            parentVm.unfoldPathToNode(vm.idVersionWithoutRoot);
            vm.removeNode(ciId);
            _.remove(vm.deployableArtifacts, { ciId: { id: ciId } });
        });
    };

    // Resources
    vm.addDeployableResource = function () {
        deploymentPackageService.openDeployableModal(vm.idVersionWithoutRoot, false, constantService.MODES.CREATION).then(function (resource) {
            vm.deployableResources.push(resource);
            deploymentPackageService.saveDeployableCi(vm.idVersionWithoutRoot, resource).then(function () {
                parentVm.unfoldPathToNode(resource.ciId.idWithoutRoot);
            });
        });
    };

    vm.editDeployableResource = function (id) {
        var model = _.find(vm.deployableResources, { ciId: { id: id } });
        deploymentPackageService.openDeployableModal(vm.idVersionWithoutRoot, false, constantService.MODES.EDITION, angular.copy(model)).then(function (resource) {
            deployableService.editDeployable(resource.ciId, resource.type, resource.form).then(function () {
                var idWithoutRoot = resource.ciId.id.substring(resource.ciId.id.indexOf('/'));
                parentVm.unfoldPathToNode(idWithoutRoot);
                angular.copy(resource, model);
            });
        });
    };

    vm.deleteDeployableResource = function (ciId) {
        deploymentPackageService.deleteResource(ciId).then(function () {
            parentVm.unfoldPathToNode(vm.idVersionWithoutRoot);
            vm.removeNode(ciId);
            _.remove(vm.deployableResources, { ciId: { id: ciId } });
        });
    };

    vm.deployableArtifacts = [];
    vm.deployableResources = [];

    var versionTypesInterval = $interval(function () {
        vm.versionTypeFormMetadata = typeSystemService.getTypeDescriptorOf(constantService.TYPE.UDM_DEPLOYMENT_PACKAGE);

        if (vm.versionTypeFormMetadata) {
            vm.versionTypeFormMetadata.properties = _(vm.versionTypeFormMetadata.properties).filter({ asContainment: false }).map(function (it) {
                if (it.kind === 'MAP_STRING_STRING') {
                    it.minVisibleRow = 4;
                }
                return it;
            }).value();

            $interval.cancel(versionTypesInterval);
        }
    }, 1000);

    vm.closePackageWindow = function () {
        return $interval(function () {
            return angular.element('#close-package-window').triggerHandler('click');
        }, 1, 1);
    };

    vm.saveVersionProperties = function () {
        delete vm.configurationError;

        var newPathAndAppName = utils.addLeadingSlashIfNotExists(vm.existingApplicationName.id);
        var appObj = ciService.extractCiStructure(constantService.ROOT.APPLICATIONS + newPathAndAppName);
        var versionObj = ciService.normalizeCiId(constantService.ROOT.APPLICATIONS, appObj.idWithoutRoot, vm.applicationVersion);
        deploymentPackageService.editCiVersion(versionObj, constantService.TYPE.UDM_DEPLOYMENT_PACKAGE, vm.versionPropertiesFormData).then(function () {
            var currentTab = $uiTabs.getActiveTab();
            currentTab.pristine = true; // TODO why DPO ?

            vm.closePackageWindow();
            switchToViewDeploymentPackage();
        }, function (err) {
            vm.configurationError = _.reduce(err.data['validation-messages'], function (acc, current) {
                return acc + current;
            });
        });
    };

    vm.editDeployablePackage = function () {
        vm.cloneVersionPropertiesFormData = angular.copy(vm.versionPropertiesFormData);
        vm.mode = tab.mode = constantService.MODES.EDITION;
    };

    vm.cancelVersionProperties = function () {
        vm.versionPropertiesFormData = vm.cloneVersionPropertiesFormData;
        vm.mode = tab.mode = constantService.MODES.VIEW;
        vm.closePackageWindow();
    };

    vm.isUploading = function () {
        return _($scope.deployableArtifacts).map(_.isNumber).some();
    };

    vm.isCreateAndConfigurationDisabled = function () {
        return vm.showDeploymentDetails || _.isEmpty(vm.versionTypeFormMetadata) || _.isEmpty(vm.existingApplicationName) || _.isEmpty(vm.applicationVersion);
    };

    $scope.$watch(function () {
        return vm.newDeploymentPackageForm && vm.newDeploymentPackageForm.$pristine;
    }, function (value) {
        if (value === false) {
            $scope.$setTabPristine(); //eslint-disable-line
        }
    });

    function switchToViewDeploymentPackage() {
        vm.mode = tab.mode = constantService.MODES.VIEW;
    }
}]);
'use strict';

angular.module('NewApplication').factory('deployableService', ["$q", "ciService", "ciResource", function ($q, ciService, ciResource) {

    var editDeployable = function editDeployable(deployableCIObj, type, data) {
        return ciResource.editCi(deployableCIObj.id, ciService.createJsonCiWithData(deployableCIObj, type, data));
    };

    return {
        editDeployable: editDeployable
    };
}]);
'use strict';

angular.module('NewApplication').controller('ApplicationsDeployableController', ["$scope", "$log", "$state", "$stateParams", "applicationsService", "typeSystemService", "constantService", "ciService", "deployableService", "$uiTabs", "ciTabsService", function ($scope, $log, $state, $stateParams, applicationsService, typeSystemService, constantService, ciService, deployableService, $uiTabs, ciTabsService) {
    $scope.$tabData.isLoaded = false;

    $scope.deployableTypeNames = typeSystemService.getDeployableTypeNames();
    $scope.deployableTypeFormValidation = {};

    if (!$scope.$tabData.initialized) {
        $scope.$tabData.deployableTypeFormMetadata = {};
        $scope.$tabData.deployableTypeFormData = {};

        applicationsService.getCi($scope.$tabData.deployableId).then(function (response) {
            var ci = response.data;

            var data = ciService.extractCiStructure(ci.id);
            $scope.$tabData.deployablePath = data.path;
            $scope.$tabData.deployableName = data.name;
            $scope.$tabData.deployableType = ci.type;

            delete ci.$token;
            delete ci.id;
            delete ci.type;

            $scope.$tabData.deployableTypeFormData = ci;
            $scope.$tabData.deployableTypeFormMetadata = typeSystemService.getTypeDescriptorOf($scope.$tabData.deployableType);
            $scope.$tabData.isLoaded = true;
        }, function (error) {
            return $log.error(error);
        });
    } else {
        $scope.$tabData.isLoaded = true;
    }

    $scope.onTypeSelect = function (typeName) {
        $scope.$tabData.deployableType = typeName;
        $scope.$tabData.deployableTypeFormData = {};
        $scope.$tabData.deployableTypeFormMetadata = typeSystemService.getTypeDescriptorOf(typeName);
    };

    $scope.saveDeployable = function (callback) {
        var deployableObj = ciService.normalizeCiId(constantService.ROOT.APPLICATIONS, $scope.$tabData.deployablePath, $scope.$tabData.deployableName);
        deployableService.editDeployable(deployableObj, $scope.$tabData.deployableType, $scope.$tabData.deployableTypeFormData).then(function () {
            $scope.$tabData.deployableId = deployableObj.id;
            if (callback) {
                callback();
            }
        });
    };

    $scope.isLoaded = function () {
        return $scope.$tabData.isLoaded;
    };

    $scope.isValid = function () {
        return !$scope.deployableTypeFormValidation.$invalid && !$scope.newDeployableForm.$invalid;
    };

    $scope.$watch(function () {
        return $scope.deployableTypeFormValidation.$pristine && $scope.newDeployableForm.$pristine;
    }, function (value) {
        if (value === false) {
            $scope.$setTabPristine();
        }
    });

    // deployable tabs
    $scope.viewDeployable = ciTabsService.viewDeployable;
    $scope.editDeployable = ciTabsService.editDeployable;
    $scope.switchToViewDeployable = ciTabsService.switchToViewDeployable;
    $scope.switchToViewDeployableWithConfirm = function () {
        if (!$uiTabs.getActiveTab().pristine) {
            $scope.confirmDialog().then(function () {
                return ciTabsService.switchToViewDeployable();
            });
        }
    };

    $scope.$tabData.initialized = true;
}]);
'use strict';

angular.module('NewApplication').directive('xlAutocomplete', function () {
    return {
        scope: {
            xlAutocomplete: '&'
        },
        require: 'ngModel',
        link: function link(scope, element, attr, ctrl) {
            element.autocomplete({
                delay: 200,
                minLength: 1,
                source: function source(params, addCandidates) {
                    scope.xlAutocomplete(params.term).then(addCandidates);
                },
                select: function select(event, ui) {
                    scope.$apply(function () {
                        ctrl.$setViewValue(ui.item.value);
                    });
                }
            });
        }
    };
});
'use strict';

angular.module('NewApplication').factory('applicationCiService', ["$q", "ciService", "ciResource", "constantService", function ($q, ciService, ciResource, constantService) {

    function createJsonCis(applicationObj, extendedData) {
        var cis = ciService.createDirectoriesJsonCis(applicationObj);
        cis.push(ciService.createJsonCiWithData(applicationObj, constantService.TYPE.UDM_APPLICATIONS, extendedData));

        return cis;
    }

    function editCiApplication(applicationCiObj, extendedData) {
        var ci = ciService.createJsonCiWithData(applicationCiObj, constantService.TYPE.UDM_APPLICATIONS, extendedData);
        return ciResource.editCi(ci.id, ci);
    }

    return {
        editCiApplication: editCiApplication,
        createJsonCis: createJsonCis
    };
}]);
'use strict';

angular.module('NewApplication').controller('ApplicationsEditController', ["$scope", "$log", "$http", "applicationsService", "applicationCiService", "ciService", "typeSystemService", "constantService", "$uiTabs", "ciTabsService", function ($scope, $log, $http, applicationsService, applicationCiService, ciService, typeSystemService, constantService, $uiTabs, ciTabsService) {
    $scope.$tabData.isLoaded = false;
    $scope.applicationMetadata = typeSystemService.getTypeDescriptorOf(constantService.TYPE.UDM_APPLICATIONS);
    $scope.applicationDynamicFormModel = {};

    if (!$scope.$tabData.initialized) {
        $scope.$tabData.application = {};

        applicationsService.getCi($scope.$tabData.applicationId).then(function (response) {
            var ci = response.data;
            var data = ciService.extractCiStructure(ci.id);
            $scope.$tabData.applicationPath = data.path;
            $scope.$tabData.applicationName = data.name;
            $scope.$tabData.application = ci;
            $scope.$tabData.isLoaded = true;
        }, function (error) {
            return $log.error(error);
        });
    } else {
        $scope.$tabData.isLoaded = true;
    }

    $scope.getDirectoriesMatching = function (applicationPathPattern) {
        return applicationsService.getExistingDirectories(applicationPathPattern);
    };

    $scope.saveApplicationAndDirectories = function (onSaveCallback) {
        var extendedData = $scope.$tabData.application;
        var applicationCIObj = ciService.normalizeCiId(constantService.ROOT.APPLICATIONS, $scope.$tabData.applicationPath, $scope.$tabData.applicationName);

        var errorFunction = function errorFunction(error) {
            $scope.$tabData.postApplicationsError = error.isApplicationError ? error.message : undefined;
            $scope.$tabData.postDirectoryError = error.isDirectoryError ? error.message : undefined;
        };

        $scope.$tabData.applicationId = applicationCIObj.id;

        applicationCiService.editCiApplication(applicationCIObj, extendedData).then(function () {
            onSaveCallback(applicationCIObj.idWithoutRoot);
        }, errorFunction);
    };

    $scope.isLoaded = function () {
        return $scope.$tabData.isLoaded;
    };

    $scope.isValid = function () {
        return !$scope.newApplicationForm.$invalid && !$scope.applicationDynamicFormModel.$invalid;
    };

    $scope.$watch(function () {
        return $scope.applicationDynamicFormModel.$pristine && $scope.newApplicationForm.$pristine;
    }, function (value) {
        if (value === false) {
            $scope.$setTabPristine();
        }
    });

    // application tabs
    $scope.viewApplication = ciTabsService.viewApplication;
    $scope.editApplication = ciTabsService.editApplication;
    $scope.switchToViewApplication = ciTabsService.switchToViewApplication;
    $scope.switchToNewVersion = ciTabsService.switchToNewVersion;
    $scope.switchToViewApplicationWithConfirm = function () {
        if (!$uiTabs.getActiveTab().pristine) {
            $scope.confirmDialog().then(function () {
                return ciTabsService.switchToViewApplication();
            });
        }
    };

    $scope.$tabData.initialized = true;
}]);
'use strict';

angular.module('NewApplication').factory('ciTabsService', ["utils", "$interval", "$uiTabs", "constantService", "confirmDialogService", function (utils, $interval, $uiTabs, constantService, confirmDialogService) {

    /**
     * Opens or replace a CI tab
     * @param targetState target tab type
     * @param options tab options
     * @param tabMode
     *  FORCE_NEW - always opens new tab,
     *  FORCE_SAME - switches the same tab if possible,
     *  AUTO - if current tab is in visual mode, switches the current tab, if not - opens new
     */
    var openOrReplaceTab = function openOrReplaceTab(targetState, options, tabMode) {
        var currentTab = $uiTabs.getActiveTab();
        if (!currentTab || !!currentTab.isWelcome || tabMode === 'FORCE_NEW') {
            $uiTabs.addTab(targetState, options);
        } else if (currentTab && (currentTab.$tabData.mode === 'visual' || tabMode === 'FORCE_SAME')) {
            $uiTabs.addTab(targetState, options, currentTab.$$id);
        }
    };

    // application tabs
    var viewApplication = function viewApplication(applicationId, forceNewTab) {
        openOrReplaceTab('applications.app.view', {
            title: utils.removePrefix(applicationId),
            $tabData: {
                applicationId: applicationId,
                mode: constantService.MODES.VIEW
            }
        }, forceNewTab);
    };

    var closePopupDialog = function closePopupDialog(tab, msg) {
        return confirmDialogService.openConfirmDialog({
            message: msg, confirm: 'Yes', reject: 'Cancel', showIf: !tab.pristine
        }).then(function () {
            return $interval(function () {
                return angular.element('#close-package-window').triggerHandler('click');
            }, 1, 1);
        });
    };

    var editApplication = function editApplication() {
        var currentTab = $uiTabs.getActiveTab();
        var $tabData = angular.copy(currentTab.$tabData);
        $tabData.mode = constantService.MODES.EDITION;
        openOrReplaceTab('applications.app.edit', {
            title: currentTab.title,
            $tabData: $tabData
        }, 'FORCE_SAME');
    };

    var switchToViewApplication = function switchToViewApplication() {
        var currentTab = $uiTabs.getActiveTab();
        openOrReplaceTab('applications.app.view', {
            title: utils.removePrefix(currentTab.$tabData.applicationId),
            $tabData: {
                applicationId: currentTab.$tabData.applicationId,
                mode: constantService.MODES.VIEW
            }
        }, 'FORCE_SAME');
    };

    // deployable tabs
    var viewDeployable = function viewDeployable(deployableId, forceNewTab) {
        openOrReplaceTab('applications.deployable.view', {
            title: utils.removePrefix(deployableId),
            $tabData: {
                deployableId: deployableId,
                mode: constantService.MODES.VIEW
            }
        }, forceNewTab);
    };

    var editDeployable = function editDeployable() {
        var currentTab = $uiTabs.getActiveTab();
        var $tabData = angular.copy(currentTab.$tabData);
        $tabData.mode = constantService.MODES.EDITION;
        openOrReplaceTab('applications.deployable.edit', {
            title: currentTab.title,
            // initialized: currentTab.initialized,
            $tabData: $tabData
        }, 'FORCE_SAME');
    };

    var switchToViewDeployable = function switchToViewDeployable() {
        var currentTab = $uiTabs.getActiveTab();
        var $tabData = angular.copy(currentTab.$tabData);
        $tabData.mode = constantService.MODES.VIEW;
        openOrReplaceTab('applications.deployable.view', {
            title: utils.removePrefix($tabData.deployableId),
            $tabData: $tabData
        }, 'FORCE_SAME');
    };

    // version deployment package
    var newDeploymentPackage = function newDeploymentPackage() {
        openOrReplaceTab('applications.deployment.package', {
            mode: constantService.MODES.CREATION
        }, 'FORCE_NEW');
    };

    // view version deployment package
    var viewDeploymentPackage = function viewDeploymentPackage(versionId) {
        openOrReplaceTab('applications.deployment.package', {
            title: utils.removePrefix(versionId),
            versionId: versionId,
            mode: constantService.MODES.VIEW
        }, 'FORCE_NEW');
    };

    // import tabs
    var importCIs = function importCIs() {
        $uiTabs.addTab('applications.import', {});
    };

    /**
     * Public API
     */
    return {
        viewApplication: viewApplication,
        editApplication: editApplication,
        switchToViewApplication: switchToViewApplication,

        viewDeployable: viewDeployable,
        editDeployable: editDeployable,
        switchToViewDeployable: switchToViewDeployable,
        closePopupDialog: closePopupDialog,

        newDeploymentPackage: newDeploymentPackage,
        viewDeploymentPackage: viewDeploymentPackage,

        importCIs: importCIs
    };
}]);
'use strict';

angular.module('NewApplication').config(["$uiTabsProvider", function ($uiTabsProvider) {

    $uiTabsProvider.tab('applications.app.view', {
        title: 'Visual application',
        templateUrl: 'application/tabs/applications/application-ci.html',
        controller: 'ApplicationsEditController'
    }).tab('applications.app.edit', {
        title: 'Edit application',
        templateUrl: 'application/tabs/applications/application-ci.html',
        controller: 'ApplicationsEditController'
    }).tab('applications.deployable.view', {
        title: 'Visual deployable',
        templateUrl: 'application/tabs/deployable/deployable.html',
        controller: 'ApplicationsDeployableController'
    }).tab('applications.deployable.edit', {
        title: 'Edit deployable',
        templateUrl: 'application/tabs/deployable/deployable.html',
        controller: 'ApplicationsDeployableController'
    }).tab('applications.import', {
        title: 'Import package',
        templateUrl: 'application/tabs/import/import.html',
        controller: 'ImportPackageController',
        controllerAs: 'vm',
        bindToController: true,
        volatile: false,
        onClose: ['ciTabsService', function (ciTabsService) {
            return function (tab) {
                return ciTabsService.closePopupDialog(tab, 'Are you sure you want to leave? Imports in progress will be cancelled.');
            };
        }]
    }).tab('applications.deployment.package', {
        title: 'New Deployment Package',
        templateUrl: 'application/tabs/deployment-package/deployment-package.html',
        controller: 'DeploymentPackageController',
        volatile: false,
        controllerAs: 'vm',
        bindToController: true
    }).welcome({
        templateUrl: 'application/tabs/welcome/welcome.html'
    }).onClose(['ciTabsService', function (ciTabsService) {
        return function (tab) {
            return ciTabsService.closePopupDialog(tab, 'Are you sure you want to leave? Your changes will be lost.');
        };
    }]);
}]);
'use strict';

angular.module('NewApplication').factory('applicationsService', ["$q", "ciResource", "applicationsResource", "directoryResource", "versionResource", "constantService", "utils", function ($q, ciResource, applicationsResource, directoryResource, versionResource, constantService, utils) {

    var applicationTypeId = 'appClass';
    var directoryTypeId = 'dirClass';
    var versionTypeId = 'versionClass';
    var deployableTypeId = 'deployableClass';

    function getCi(ciId) {
        return ciResource.get(ciId);
    }

    function getExistingDirectories(namePattern) {
        return directoryResource.findAllDirectoriesByAncestor(constantService.ROOT.APPLICATIONS, namePattern).then(utils.getCiIdsWithoutPrefix);
    }

    function findAllCis(type, namePattern) {
        return ciResource.findAllCis(type, namePattern);
    }

    function addCandidatesApplication(meta, params) {
        return applicationsResource.findAllApplications(params.term).then(function (response) {
            return _.map(response.data, function (ci) {
                ci.id = utils.removePrefix(ci.ref);
                return ci;
            });
        });
    }

    function ciExists(ciName) {
        return ciResource.ciExists(constantService.ROOT.APPLICATIONS + '/' + ciName);
    }

    function addTypeToReturnedData(responseData, classType) {
        _.forEach(responseData, function (it) {
            it.classType = classType;
        });
        return responseData;
    }

    function parseDeployableRefToNodeData(deployableRef) {
        return { ref: deployableRef, classType: deployableTypeId };
    }

    function parseDataToSingleNode(data) {
        var path = data.ref;
        var classType = data.classType;
        var labels = path.split('/');
        var id = 'id_' + path;
        var idDom = id.replace(new RegExp('/', 'g'), '-');
        idDom = idDom.replace(new RegExp(' ', 'g'), '-');
        var node = { id: 'id_' + path, path: path, label: _.last(labels), classType: classType, idDom: idDom };

        if (classType !== deployableTypeId) {
            node.children = [];
        }

        return node;
    }

    function findCIStructureByParent(parentPath, parentType) {
        var parent = parentPath ? parentPath : constantService.ROOT.APPLICATIONS;
        var promises = [];
        if (!parentType || parentType === directoryTypeId) {
            // we look for root nodes or children of a directory.
            var directoriesPromise = directoryResource.findAllDirectoriesByParent(parent).then(function (response) {
                return addTypeToReturnedData(response.data, directoryTypeId);
            });
            var applicationsPromise = applicationsResource.findAllApplicationsByParent(parent).then(function (response) {
                return addTypeToReturnedData(response.data, applicationTypeId);
            });
            promises.push(directoriesPromise);
            promises.push(applicationsPromise);
        } else if (parentType && parentType === applicationTypeId) {
            var versionsPromise = versionResource.findAllVersionsByParent(parent).then(function (response) {
                return addTypeToReturnedData(response.data, versionTypeId);
            });
            promises.push(versionsPromise);
        } else if (parentType && parentType === versionTypeId) {
            var deployablePromise = ciResource.get(parent).then(function (response) {
                return _.map(response.data.deployables, parseDeployableRefToNodeData);
            });
            promises.push(deployablePromise);
        }

        return $q.all(promises).then(function (allData) {
            var structureData = [];
            _.forEach(allData, function (data) {
                structureData = _.union(structureData, data);
            });
            structureData = _.map(structureData, parseDataToSingleNode);

            return structureData;
        });
    }

    return {
        applicationTypeId: applicationTypeId,
        directoryTypeId: directoryTypeId,
        versionTypeId: versionTypeId,
        deployableTypeId: deployableTypeId,

        getCi: getCi,
        findAllCis: findAllCis,
        ciExists: ciExists,
        getExistingDirectories: getExistingDirectories,
        addCandidatesApplication: addCandidatesApplication,
        findCIStructureByParent: findCIStructureByParent,
        addTypeToReturnedData: addTypeToReturnedData
    };
}]);
'use strict';

angular.module('NewApplication').factory('applicationsResource', ["$http", "ciResource", "constantService", "repositoryQueryBuilder", function ($http, ciResource, constantService, repositoryQueryBuilder) {

    return {
        findAllApplications: function findAllApplications(namePattern) {
            var query = repositoryQueryBuilder.newQueryBuilder().withType(constantService.TYPE.UDM_APPLICATIONS).withNamePattern(namePattern).query();
            return ciResource.queryRepository(query);
        },
        findAllApplicationsByParent: function findAllApplicationsByParent(parentId, namePattern) {
            var query = repositoryQueryBuilder.newQueryBuilder().withType(constantService.TYPE.UDM_APPLICATIONS).withNamePattern(namePattern).withParent(parentId).query();
            return ciResource.queryRepository(query);
        }
    };
}]);
'use strict';

angular.module('NewApplication').controller('ApplicationsManagerController', ["$interval", "$http", "$state", "applicationsService", "treeService", "ciTabsService", "constantService", "confirmDialogService", function ($interval, $http, $state, applicationsService, treeService, ciTabsService, constantService, confirmDialogService) {

    var vm = this;

    vm.applications = [];
    vm.currentNode = {};

    vm.selectNode = function (node) {
        return vm.currentNode.path = node.path;
    };

    vm.unfoldPathToNode = function (targetNodePath) {
        var nodeLabelsInPath = targetNodePath.split('/');
        if (targetNodePath[0] === '/') {
            nodeLabelsInPath.splice(0, 1);
        }
        var parentNode = { path: constantService.ROOT.APPLICATIONS, children: vm.applications, classType: 'dirClass' };
        treeService.unfoldNode(nodeLabelsInPath, 0, parentNode, vm.selectNode, applicationsService.findCIStructureByParent);
    };

    vm.getRootCIs = function () {
        applicationsService.findCIStructureByParent().then(function (rootCIs) {
            return vm.applications = rootCIs;
        });
    };

    vm.findNodeChildren = function (node) {
        return applicationsService.findCIStructureByParent(node.path, node.classType);
    };

    vm.removeNode = function (id) {
        var nodeLabelsInPath = id.split('/');
        treeService.removeNode(nodeLabelsInPath.splice(1), vm.applications);
    };

    vm.treeOptions = {
        clickAction: _.noop,
        dblClickAction: _.noop
    };

    vm.getRootCIs();

    vm.newDeploymentPackage = ciTabsService.newDeploymentPackage;
    vm.importCIs = ciTabsService.importCIs;

    vm.confirmDialog = function () {
        return confirmDialogService.openConfirmDialog({
            message: 'Are you sure you want to leave? Any changes will be lost.',
            confirm: 'Yes', reject: 'Cancel'
        }).then(function () {
            return $interval(function () {
                return angular.element('#close-package-window').triggerHandler('click');
            }, 1, 1);
        });
    };
}]);
//# sourceMappingURL=application.js.map
