'use strict';

angular.module('xl.online.commons', ['ngLocalize', 'ngSanitize']);
'use strict';

angular.module('xl.online.commons').factory('treeService', function (directoryResource, deploymentResource, versionResource, ciResource, environmentsResource) {
    var applicationTypeId = 'appClass';
    var directoryTypeId = 'dirClass';
    var versionTypeId = 'versionClass';
    var deployableTypeId = 'deployableClass';
    var environmentTypeId = 'envClass';
    var deployedApplicationTypeId = 'deployedAppClass';
    var deployedContainerTypeId = 'deployedContainerClass';
    var deployedTypeId = 'deployedClass';
    var artifactTypeId = 'artifactClass';
    var environmentTypeClass = 'environmentClass';

    var sortCisByLabel = function sortCisByLabel(a, b) {
        if (a.label < b.label) {
            return -1;
        } else if (a.label === b.label) {
            return 0;
        } else {
            return 1;
        }
    };

    var completeCiArray = function completeCiArray(targetArray, updatingValues) {
        var existingLabels = _.map(targetArray, 'label');
        var retainedValues = _.reject(updatingValues, function (value) {
            return _.includes(existingLabels, value.label);
        });
        _.forIn(retainedValues, function (node) {
            return targetArray.push(node);
        });
    };

    var unfoldNode = function unfoldNode(labelsToTargetNode, labelIdx, parentNode, selectCallback, findStructureByParent) {

        var openNode = function openNode(node) {
            if (labelIdx === labelsToTargetNode.length - 1) {
                // we just found the target node to select
                node.from = 'create';
                selectCallback(node);
            } else if (node.children && node.children.length > 0) {
                node.open = true;
                if (labelIdx + 1 < labelsToTargetNode.length) {
                    unfoldNode(labelsToTargetNode, labelIdx + 1, node, selectCallback, findStructureByParent);
                }
            } else {
                // we load the node's children
                findStructureByParent(node.path, node.classType).then(function (childrenNodes) {
                    if (childrenNodes && childrenNodes.length > 0) {
                        if (!node.children) {
                            node.children = [];
                        }
                        completeCiArray(node.children, childrenNodes);
                        node.children.sort(sortCisByLabel);
                        node.open = true;
                        if (labelIdx + 1 < labelsToTargetNode.length) {
                            unfoldNode(labelsToTargetNode, labelIdx + 1, node, selectCallback, findStructureByParent);
                        }
                    }
                });
            }
        };

        var reloadDirectChildren = function reloadDirectChildren() {
            findStructureByParent(parentNode.path, parentNode.classType).then(function (childrenNodes) {
                if (childrenNodes && childrenNodes.length > 0) {
                    if (angular.equals(parentNode.children, childrenNodes)) {
                        throw new Error('even after refreshing the children nodes of a parent, no matching path could be found');
                    }
                    completeCiArray(parentNode.children, childrenNodes);
                    parentNode.children.sort(sortCisByLabel);
                    unfoldNode(labelsToTargetNode, labelIdx, parentNode, selectCallback, findStructureByParent);
                } else {
                    throw new Error('even after refreshing the children nodes of a parent, no matching path could be found');
                }
            });
        };

        var currentDepthLabel = labelsToTargetNode[labelIdx];
        var matchingNodes = _.filter(parentNode.children, { label: currentDepthLabel });

        if (matchingNodes.length > 1) {
            throw new Error('there should not be duplicated node names with the same parent node');
        } else if (matchingNodes.length === 1) {
            // there is a single matching node already loaded: we open it
            openNode(matchingNodes[0]);
        } else {
            // no matching node: we should reload the all the direct children of the parent node
            reloadDirectChildren();
        }
    };

    var removeNode = function removeNode(ids, nodes) {
        var removeNodeInner = function removeNodeInner(fromNodes, index) {
            var id = ids[index];
            if (index === ids.length - 1) {
                _.remove(fromNodes, { label: id });
            } else {
                var node = _.find(fromNodes, { label: id });
                var children = node ? node.children : false;
                removeNodeInner(children, index + 1);
            }
        };

        if (ids && ids.length > 0) {
            removeNodeInner(nodes, 0);
        }
    };

    var addTypeToReturnedData = function addTypeToReturnedData(responseData, classType) {
        _.forEach(responseData, function (it) {
            return it.classType = classType;
        });
        return responseData;
    };

    var findDirectoriesByParent = function findDirectoriesByParent(parentPath) {
        return directoryResource.findAllDirectoriesByParent(parentPath).then(function (response) {
            return addTypeToReturnedData(response.data, directoryTypeId);
        });
    };

    var findApplicationsByParent = function findApplicationsByParent(parentPath) {
        return deploymentResource.findAllApplicationsByParent(parentPath).then(function (response) {
            return addTypeToReturnedData(response.data, applicationTypeId);
        });
    };

    var findVersionsByParent = function findVersionsByParent(parentPath) {
        return versionResource.findAllVersionsByParent(parentPath).then(function (response) {
            return addTypeToReturnedData(response.data, versionTypeId);
        });
    };

    var findDeployablesByParent = function findDeployablesByParent(parentPath) {
        return ciResource.get(parentPath).then(function (response) {
            return _.map(response.data.deployables, function (deployableRef) {
                return { ref: deployableRef, classType: deployableTypeId };
            });
        });
    };

    var findCisByParent = function findCisByParent(parentPath) {
        return deploymentResource.findAllCIs(parentPath).then(function (response) {
            return _.map(response.data, function (ci) {
                return { ref: ci.ref, classType: deployableTypeId };
            });
        });
    };

    var findEnvironmentsByParent = function findEnvironmentsByParent(parentPath) {
        return environmentsResource.findAllEnvironmentsByParent(parentPath).then(function (response) {
            return addTypeToReturnedData(response.data, environmentTypeId);
        });
    };

    var findDeployedApplicationsByParent = function findDeployedApplicationsByParent(parentPath) {
        return deploymentResource.findAllDeployedApplicationsByParent(parentPath).then(function (response) {
            return addTypeToReturnedData(response.data, deployedApplicationTypeId);
        });
    };

    var loadCiListAsTreeNodes = function loadCiListAsTreeNodes(ids, typeId) {
        return ciResource.getCisByIDList(ids).then(function (cis) {
            _.forEach(cis.data, function (application) {
                application.ref = application.id;
            });
            return addTypeToReturnedData(cis.data, typeId);
        });
    };

    return {
        applicationTypeId: applicationTypeId,
        directoryTypeId: directoryTypeId,
        versionTypeId: versionTypeId,
        deployableTypeId: deployableTypeId,
        environmentTypeId: environmentTypeId,
        deployedApplicationTypeId: deployedApplicationTypeId,
        deployedContainerTypeId: deployedContainerTypeId,
        deployedTypeId: deployedTypeId,
        artifactTypeId: artifactTypeId,
        environmentTypeClass: environmentTypeClass,

        unfoldNode: unfoldNode,
        removeNode: removeNode,
        findDirectoriesByParent: findDirectoriesByParent,
        findApplicationsByParent: findApplicationsByParent,
        findVersionsByParent: findVersionsByParent,
        findDeployablesByParent: findDeployablesByParent,
        findEnvironmentsByParent: findEnvironmentsByParent,
        findDeployedApplicationsByParent: findDeployedApplicationsByParent,

        loadCiListAsTreeNodes: loadCiListAsTreeNodes,
        addTypeToReturnedData: addTypeToReturnedData,
        findCisByParent: findCisByParent
    };
});
'use strict';

angular.module('xl.online.commons').directive('scrollTo', function () {
    return {
        restrict: 'A',
        scope: {
            scrollToMe: '='
        },
        link: function link($scope, $element) {
            if ($scope.scrollToMe) {
                angular.element('.content').animate({ scrollTop: $element.offset().top }, 100, 'linear');
                $scope.scrollToMe = false;
            }
        }
    };
});
'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var TaskTransformer = function () {
    function TaskTransformer() {
        _classCallCheck(this, TaskTransformer);
    }

    _createClass(TaskTransformer, [{
        key: '$get',
        value: function $get(stepTransformer) {

            var excludeSingleBlocks = function excludeSingleBlocks(apiEntity) {
                if (apiEntity.block.blocks && apiEntity.block.blocks.length === 1) {
                    return createBlockWithChildren(apiEntity.block.blocks[0]);
                } else {
                    return createBlockWithChildren(apiEntity.block);
                }
            };

            var createBlock = function createBlock(apiEntity) {
                return {
                    id: 'id_' + apiEntity.id,
                    idDom: 'id_dom_' + apiEntity.id,
                    classType: '',
                    label: apiEntity.description,
                    path: apiEntity.id,
                    state: apiEntity.state,
                    hasSteps: apiEntity.hasSteps
                };
            };

            var createBlockWithChildren = function createBlockWithChildren(apiEntity) {
                if (apiEntity.block) {
                    return excludeSingleBlocks(apiEntity);
                }

                var treeNode = createBlock(apiEntity);

                if (apiEntity.hasSteps) {
                    treeNode.children = apiEntity.steps ? stepTransformer.transformSteps(apiEntity.steps, apiEntity.id) : [];
                }

                if (apiEntity.blocks) {
                    treeNode.children = _.map(apiEntity.blocks, function (block) {
                        return createBlockWithChildren(block);
                    });
                }
                return treeNode;
            };

            var transformTask = function transformTask(taskApi) {
                return {
                    id: taskApi.id,
                    state: taskApi.state,
                    rootBlocks: _.map(taskApi.block.blocks, function (block) {
                        return createBlockWithChildren(block);
                    })
                };
            };

            return {
                transformTask: transformTask
            };
        }
    }]);

    return TaskTransformer;
}();

angular.module('xl.online.commons').provider('taskTransformer', TaskTransformer);
'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var StepTransformer = function () {
    function StepTransformer() {
        _classCallCheck(this, StepTransformer);
    }

    _createClass(StepTransformer, [{
        key: '$get',
        value: function $get() {
            var parseStep = function parseStep(blockPath, stepIndex, step) {
                var path = blockPath + '_' + (stepIndex + 1);
                return {
                    id: 'id_' + path,
                    idDom: 'id_dom_' + path,
                    classType: 'stepNoPreviewClass',
                    path: path,
                    blockPath: blockPath,
                    label: step.description,
                    state: step.state,
                    stepNode: true
                };
            };

            var transformSteps = function transformSteps(steps, blockPath) {
                return _.map(steps, function (step, index) {
                    return parseStep(blockPath, index, step);
                });
            };

            return {
                transformSteps: transformSteps
            };
        }
    }]);

    return StepTransformer;
}();

angular.module('xl.online.commons').provider('stepTransformer', StepTransformer);
'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var DeploymentRequestTransformer = function () {
    function DeploymentRequestTransformer() {
        _classCallCheck(this, DeploymentRequestTransformer);
    }

    _createClass(DeploymentRequestTransformer, [{
        key: '$get',
        value: function $get() {
            var transform = function transform(object) {
                return {
                    id: object.id,
                    deployeds: object.deployeds ? object.deployeds : [],
                    deployables: object.deployables ? object.deployables : [],
                    containers: object.containers ? object.containers : [],
                    type: object.type,
                    application: {
                        id: object.application.id,
                        optimizePlan: object.optimizePlan ? object.optimizePlan : true,
                        type: object.application.type,
                        version: object.application.version,
                        environment: object.application.environment
                    }
                };
            };

            return {
                transform: transform
            };
        }
    }]);

    return DeploymentRequestTransformer;
}();

angular.module('xl.online.commons').provider('deploymentRequestTransformer', DeploymentRequestTransformer);
'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var BlockTransformer = function () {
    function BlockTransformer() {
        _classCallCheck(this, BlockTransformer);
    }

    _createClass(BlockTransformer, [{
        key: '$get',
        value: function $get(stepTransformer) {

            var transformBlock = function transformBlock(blockApi) {
                return {
                    path: blockApi.id,
                    state: blockApi.state,
                    hasSteps: blockApi.hasSteps,
                    steps: stepTransformer.transformSteps(blockApi.steps, blockApi.id)
                };
            };

            return {
                transformBlock: transformBlock
            };
        }
    }]);

    return BlockTransformer;
}();

angular.module('xl.online.commons').provider('blockTransformer', BlockTransformer);
'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var VersionResource = function () {
    function VersionResource() {
        _classCallCheck(this, VersionResource);
    }

    _createClass(VersionResource, [{
        key: '$get',
        value: function $get(ciResource, repositoryQueryBuilder) {

            var findAllVersions = function findAllVersions(namePattern) {
                var query = repositoryQueryBuilder.newQueryBuilder().withType('udm.DeploymentPackage').withNamePattern(namePattern).query();
                return ciResource.queryRepository(query);
            };

            var findAllVersionsByParent = function findAllVersionsByParent(parentId) {
                var query = repositoryQueryBuilder.newQueryBuilder().withType('udm.Version').withParent(parentId).query();
                return ciResource.queryRepository(query);
            };

            return {
                findAllVersions: findAllVersions,
                findAllVersionsByParent: findAllVersionsByParent
            };
        }
    }]);

    return VersionResource;
}();

angular.module('xl.online.commons').provider('versionResource', VersionResource);
'use strict';

angular.module('xl.online.commons').factory('taskResource', function (xlHttp, taskTransformer, blockTransformer) {

    var abort = function abort(taskId) {
        return xlHttp.httpPost({ endpoint: 'tasks/v2/' + taskId + '/abort' });
    };
    var addPauseStep = function addPauseStep(taskId, stepId) {
        return xlHttp.httpPost({ endpoint: 'tasks/v2/' + taskId + '/add-pause/' + stepId });
    };
    var archive = function archive(taskId) {
        return xlHttp.httpPost({ endpoint: 'tasks/v2/' + taskId + '/archive' });
    };
    var skip = function skip(taskId, paths) {
        return xlHttp.httpPost({ endpoint: 'tasks/v2/' + taskId + '/skip', data: paths });
    };
    var unskip = function unskip(taskId, paths) {
        return xlHttp.httpPost({ endpoint: 'tasks/v2/' + taskId + '/unskip', data: paths });
    };
    var blockWithSteps = function blockWithSteps(taskId, blockPath) {
        return xlHttp.httpGet({
            endpoint: 'tasks/v2/' + taskId + '/block/' + blockPath + '/step', transform: blockTransformer.transformBlock
        });
    };
    var cancel = function cancel(taskId) {
        return xlHttp.httpDelete({ endpoint: 'tasks/v2/' + taskId });
    };
    var getTaskStepLog = function getTaskStepLog(taskId, stepId) {
        return xlHttp.httpGet({ endpoint: 'tasks/v2/' + taskId + '/step/' + stepId });
    };
    var start = function start(taskId) {
        return xlHttp.httpPost({ endpoint: 'tasks/v2/' + taskId + '/start' });
    };
    var stop = function stop(taskId) {
        return xlHttp.httpPost({ endpoint: 'tasks/v2/' + taskId + '/stop' });
    };
    var task = function task(taskId) {
        return xlHttp.httpGet({ endpoint: 'tasks/v2/' + taskId, transform: taskTransformer.transformTask });
    };

    return {
        abort: abort,
        archive: archive,
        skip: skip,
        unskip: unskip,
        addPauseStep: addPauseStep,
        blockWithSteps: blockWithSteps,
        cancel: cancel,
        getTaskStepLog: getTaskStepLog,
        start: start,
        stop: stop,
        task: task
    };
});
'use strict';

angular.module('xl.online.commons').factory('repositoryQueryBuilder', function () {
    var QueryBuilder = function QueryBuilder() {

        var query = { params: { resultsPerPage: -1 } };

        this.withType = function (type) {
            if (type) {
                query.params.type = type;
            }
            return this;
        };

        this.withParent = function (parent) {
            if (parent) {
                query.params.parent = parent;
            }
            return this;
        };

        this.withAncestor = function (ancestor) {
            if (ancestor) {
                query.params.ancestor = ancestor;
            }
            return this;
        };

        this.withNamePattern = function (namePattern) {
            if (namePattern) {
                query.params.namePattern = '%' + namePattern + '%';
            }
            return this;
        };

        this.startingWithNamePattern = function (namePattern) {
            if (namePattern) {
                query.params.namePattern = namePattern + '%';
            }
            return this;
        };

        this.query = function () {
            return query;
        };
    };

    var newQueryBuilder = function newQueryBuilder() {
        return new QueryBuilder();
    };

    return {
        newQueryBuilder: newQueryBuilder
    };
});
'use strict';

angular.module('xl.online.commons').factory('environmentsResource', function ($http, ciResource, constantService, repositoryQueryBuilder) {
    return {
        findAllEnvironmentsByParent: function findAllEnvironmentsByParent(parentId, namePattern) {
            var query = repositoryQueryBuilder.newQueryBuilder().withType(constantService.TYPE.UDM_ENVIRONMENT).withNamePattern(namePattern).withParent(parentId).query();
            return ciResource.queryRepository(query);
        }
    };
});
'use strict';

angular.module('xl.online.commons').factory('directoryResource', function ($http, ciResource, repositoryQueryBuilder) {

    return {
        findAllDirectories: function findAllDirectories(namePattern) {
            var query = repositoryQueryBuilder.newQueryBuilder().withType('core.Directory').withNamePattern(namePattern).query();
            return ciResource.queryRepository(query);
        },
        findAllDirectoriesByParent: function findAllDirectoriesByParent(parentId, namePattern) {
            var query = repositoryQueryBuilder.newQueryBuilder().withType('core.Directory').withNamePattern(namePattern).withParent(parentId).query();
            return ciResource.queryRepository(query);
        },
        findAllDirectoriesByAncestor: function findAllDirectoriesByAncestor(ancestorId, namePattern) {
            var query = repositoryQueryBuilder.newQueryBuilder().withType('core.Directory').withNamePattern(namePattern).withAncestor(ancestorId).query();
            return ciResource.queryRepository(query);
        }
    };
});
'use strict';

angular.module('xl.online.commons').factory('deploymentResource', function ($http, taskTransformer, ciResource, xlDeployServicesRoot, constantService, repositoryQueryBuilder, xlHttp, deploymentRequestTransformer) {

    return {
        createTask: function createTask(deployment) {
            return $http.post(xlDeployServicesRoot + 'deployment', deployment);
        },
        findAllCIs: function findAllCIs(parentId) {
            var query = repositoryQueryBuilder.newQueryBuilder().withParent(parentId).query();
            return ciResource.queryRepository(query);
        },
        findAllApplications: function findAllApplications(namePattern) {
            var query = repositoryQueryBuilder.newQueryBuilder().withType(constantService.TYPE.UDM_APPLICATIONS).withNamePattern(namePattern).query();
            return ciResource.queryRepository(query);
        },
        findAllApplicationsByParent: function findAllApplicationsByParent(parentId, namePattern) {
            var query = repositoryQueryBuilder.newQueryBuilder().withType(constantService.TYPE.UDM_APPLICATIONS).withNamePattern(namePattern).withParent(parentId).query();
            return ciResource.queryRepository(query);
        },
        findAllDeployedApplicationsByParent: function findAllDeployedApplicationsByParent(parentId) {
            var query = repositoryQueryBuilder.newQueryBuilder().withType(constantService.TYPE.UDM_DEPLOYED_APPLICATION).withParent(parentId).query();
            return ciResource.queryRepository(query);
        },
        generateSelectedDeployeds: function generateSelectedDeployeds(deploymentRequest, selectedIds) {
            return xlHttp.httpPost({
                endpoint: 'deployment/generate/selected/deployables',
                data: { 'mapping-for-deployables': selectedIds, deployment: deploymentRequest }
            });
        },
        prepareAutoDeployeds: function prepareAutoDeployeds(deployment) {
            return xlHttp.httpPost({
                endpoint: 'deployment/prepare/deployeds',
                data: deployment,
                transform: deploymentRequestTransformer.transform
            });
        },
        prepareInitial: function prepareInitial(application, environment) {
            return xlHttp.httpGet({
                endpoint: 'deployment/prepare/initial',
                params: { environment: environment, version: application },
                transform: deploymentRequestTransformer.transform
            });
        },
        prepareUpdate: function prepareUpdate(deployedApplication, version) {
            return xlHttp.httpGet({
                endpoint: 'deployment/prepare/update',
                params: { deployedApplication: deployedApplication, version: version },
                transform: deploymentRequestTransformer.transform
            });
        },
        previewBlock: function previewBlock(deployment) {
            return xlHttp.httpPost({
                endpoint: 'deployment/previewblock',
                data: deployment,
                transform: taskTransformer.transformTask
            });
        },
        rollback: function rollback(taskId) {
            return $http.post(xlDeployServicesRoot + 'deployment/rollback/' + taskId);
        },
        validate: function validate(deployment) {
            return $http.post(xlDeployServicesRoot + 'deployment/validate', deployment, {
                headers: {
                    'Accept-Type': 'application/json'
                }
            });
        },
        unDeploy: function unDeploy(deployedAppId) {
            return $http.get(xlDeployServicesRoot + 'deployment/prepare/undeploy', { params: { deployedApplication: deployedAppId } });
        }
    };
});
'use strict';

angular.module('xl.online.commons').factory('ciResource', function ($http, xlDeployServicesRoot, repositoryQueryBuilder, httpMultipartService) {

    var queryRepository = function queryRepository(query) {
        return $http.get(xlDeployServicesRoot + 'repository/query', query);
    };

    return {
        queryRepository: queryRepository,
        get: function get(ciId) {
            return $http.get(xlDeployServicesRoot + 'repository/ci/' + ciId);
        },
        saveAllCis: function saveAllCis(cis) {
            return $http.post(xlDeployServicesRoot + 'repository/cis', cis);
        },
        saveCi: function saveCi(ci) {
            return $http.post(xlDeployServicesRoot + 'repository/ci/' + ci.id, ci);
        },
        saveCiMultipart: function saveCiMultipart(ci, file) {
            var url = xlDeployServicesRoot + 'repository/ci/' + ci.id;
            return httpMultipartService.save(url, {
                fileData: file,
                filename: file.name,
                configurationItem: new Blob([angular.toJson(ci)], { type: 'application/json' })
            });
        },
        findAllCis: function findAllCis(type, namePattern) {
            var query = repositoryQueryBuilder.newQueryBuilder().withType(type).withNamePattern(namePattern).query();
            return queryRepository(query);
        },
        ciExists: function ciExists(ciName) {
            return $http.get(xlDeployServicesRoot + 'repository/exists/' + ciName);
        },
        editCi: function editCi(ciName, ci) {
            return $http({
                url: xlDeployServicesRoot + 'repository/ci/' + ciName,
                dataType: 'json',
                method: 'PUT',
                data: ci,
                headers: {
                    'Content-Type': 'application/json',
                    'Accept-Type': 'application/json'
                }
            });
        },
        deleteCi: function deleteCi(ciId) {
            return $http.delete(xlDeployServicesRoot + 'repository/ci/' + ciId);
        },
        findAllCisByAncestor: function findAllCisByAncestor(ancestorId) {
            var query = repositoryQueryBuilder.newQueryBuilder().withAncestor(ancestorId).query();
            return queryRepository(query);
        },
        findAllCisByType: function findAllCisByType(type, parentId) {
            var query = repositoryQueryBuilder.newQueryBuilder().withType(type).withParent(parentId).query();
            return queryRepository(query);
        },
        getCisByIDList: function getCisByIDList(ciIds) {
            return $http.post(xlDeployServicesRoot + 'repository/cis/read', ciIds);
        }
    };
});
'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var XlHttpService = function () {
    function XlHttpService() {
        _classCallCheck(this, XlHttpService);
    }

    _createClass(XlHttpService, [{
        key: '$get',
        value: function $get($http, xlDeployServicesRoot) {
            var resolveTransformers = function resolveTransformers(mappingInput) {
                var resolveMapping = function resolveMapping() {
                    if (_.isFunction(mappingInput)) {
                        return { 200: mappingInput };
                    } else {
                        return mappingInput;
                    }
                };

                var transformerFunction = function transformerFunction(mapping) {
                    return function (data, headers, status) {
                        return _.isUndefined(mapping[status]) ? data : mapping[status](data, headers, status);
                    };
                };

                var mapping = resolveMapping();
                return _.flatten([$http.defaults.transformResponse, transformerFunction(mapping)]);
            };

            var request = function request(options) {
                var transformerMapping = options.transform || {};
                return $http({
                    method: options.method,
                    url: '' + xlDeployServicesRoot + options.endpoint,
                    params: options.params,
                    data: options.data,
                    transformResponse: resolveTransformers(transformerMapping)
                });
            };

            var httpGet = function httpGet(options) {
                return request(angular.extend(options, { method: 'GET' }));
            };
            var httpPost = function httpPost(options) {
                return request(angular.extend(options, { method: 'POST' }));
            };
            var httpDelete = function httpDelete(options) {
                return request(angular.extend(options, { method: 'DELETE' }));
            };

            return {
                httpGet: httpGet,
                httpPost: httpPost,
                httpDelete: httpDelete
            };
        }
    }]);

    return XlHttpService;
}();

angular.module('xl.online.commons').provider('xlHttp', XlHttpService);
'use strict';

angular.module('xl.online.commons').config(function ($httpProvider) {
    /*eslint no-undef: 0*/
    var flexApp = parent.document.getElementById('flexApplication');
    if (flexApp) {
        $httpProvider.defaults.headers.common.Authorization = flexApp.getBasicAuth();
    }
    $httpProvider.defaults.headers.post['Content-Type'] = 'application/json';
});
'use strict';

angular.module('xl.online.commons').factory('httpMultipartService', function ($http, $q) {
    var save = function save(url, data) {
        var fd = new FormData();
        _.forEach(data, function (value, key) {
            fd.append(key, value);
        });

        var deferred = $q.defer();

        $.ajax({
            url: url,
            type: 'POST',
            data: fd,
            contentType: false,
            processData: false,
            xhr: function xhr() {
                var _xhr = $.ajaxSettings.xhr();
                if (_xhr.upload) {
                    _xhr.upload.addEventListener('loadstart', function () {
                        deferred.notify({ status: 'loadstart' });
                    }, false);

                    _xhr.upload.addEventListener('loadend', function () {
                        deferred.notify({ status: 'loadend' });
                    }, false);

                    // onabort: onerror: onload ontimeout

                    _xhr.upload.addEventListener('progress', function (evt) {
                        if (evt.lengthComputable) {
                            deferred.notify({
                                status: 'progress',
                                loaded: evt.loaded,
                                total: evt.total,
                                abort: function abort() {
                                    return _xhr.abort();
                                }
                            });
                        }
                    }, false);
                }
                return _xhr;
            },
            complete: function complete(jqXHR, textStatus) {
                deferred.resolve(textStatus);
            },
            error: function error(jqXHR, textStatus, errorThrown) {
                deferred.reject(errorThrown);
            },
            beforeSend: function beforeSend(xhr) {
                if ($http.defaults.headers.common.Authorization) {
                    xhr.setRequestHeader('Authorization', $http.defaults.headers.common.Authorization);
                }
            }
        });

        return deferred.promise;
    };

    return {
        save: save
    };
});
'use strict';

angular.module('xl.online.commons').config(function ($httpProvider) {
    $httpProvider.interceptors.push(function ($q, xlNotify) {
        var errorService = angular.injector(['xl.online.commons.utils']).get('errorService');
        return {
            'responseError': function responseError(rejection) {
                if (rejection.status >= 400) {
                    if (errorService.isGenericError(rejection.data)) {
                        xlNotify.notify({ message: rejection.data, title: rejection.statusText }, 'ERROR');
                    }
                    rejection.xlPreventDefault = true;
                }
                return $q.reject(rejection);
            }
        };
    });
});
'use strict';

angular.module('xl.online.commons').config(['$httpProvider', function ($httpProvider) {

    /**
     * Adding an additional params to every request to fix the cache issue on IE 8
     */
    $httpProvider.interceptors.push(function () {
        return {
            request: function request(config) {
                if (config.method === 'GET') {
                    if (config.params) {
                        config.params.cacheBuster = new Date().getTime();
                    } else if (config.url.indexOf('.html') === -1) {
                        config.params = {
                            cacheBuster: new Date().getTime()
                        };
                    }
                }
                return config;
            }
        };
    });
}]);
'use strict';

angular.module('xl.online.commons').directive('xlStickyParent', function () {
    return {
        restrict: 'A',
        controller: function controller($element) {
            this.getWidth = function () {
                return $element.width();
            };

            this.isDisplayBlock = function () {
                return $element.css('display') === 'block';
            };
        }
    };
});

angular.module('xl.online.commons').directive('xlSticky', function ($window) {
    return {
        require: '^xlStickyParent',
        restrict: 'A',
        link: function link(scope, element, attrs, parentCtrl) {
            var position = element.css('position');

            function checkIfShouldStick() {
                if (parentCtrl.isDisplayBlock()) {
                    element.css({
                        position: position,
                        width: 'auto'
                    });
                } else {
                    element.css({
                        position: 'fixed',
                        width: parentCtrl.getWidth()
                    });
                }
            }

            angular.element($window).on('resize', function () {
                checkIfShouldStick();
            });

            checkIfShouldStick();
        }
    };
});
'use strict';

angular.module('xl.online.commons').directive('onSingleClick', function ($parse, $interval) {
    return {
        restrict: 'A',
        link: function link(scope, element, attr) {
            var fn = $parse(attr.onSingleClick);
            var delay = 300,
                clicks = 0,
                interval = null;
            element.on('click', function (event) {
                clicks++; //count clicks
                if (clicks === 1) {
                    interval = $interval(function () {
                        fn(scope, { $event: event });
                        clicks = 0; //after action performed, reset counter
                    }, delay);
                } else {
                    $interval.cancel(interval); //prevent single-click action
                    clicks = 0; //after action performed, reset counter
                }
            });
        }
    };
});
'use strict';

angular.module('xl.online.commons').factory('dipHandlers', function (ciResource, utils) {

    // Handlers
    var ciHandler = function ciHandler(referencedType, term) {
        return ciResource.findAllCis(referencedType, utils.getBaseName(term)).then(function (response) {
            return _.map(response.data, function (item) {
                item.id = item.ref;
                delete item.ref;
                return item;
            });
        });
    };

    var ciFormHandler = function ciFormHandler(metadata, options) {
        return ciHandler(metadata.referencedType, options.term);
    };

    return {
        formHandlers: {
            CI: { addCandidates: ciFormHandler },
            SET_OF_CI: { addCandidates: ciFormHandler },
            LIST_OF_CI: { addCandidates: ciFormHandler }
        },
        ciHandler: ciHandler
    };
});
'use strict';

angular.module('xl.online.commons').directive('dialogPopup', [function () {
    return {
        restrict: 'A',
        scope: {
            options: '='
        },
        templateUrl: 'common/dialog/dialog.html',
        replace: true,
        controller: _.noop
    };
}]);
'use strict';

angular.module('xl.online.commons').factory('confirmDialogService', function ($q, $xlModal) {
    var showDialog = function showDialog(options) {
        return $xlModal.open({
            templateUrl: 'common/confirm-dialog/confirm-dialog.html',
            windowClass: 'xl-confirm-modal',
            controller: function controller($scope, $xlModalInstance) {
                $scope.message = options.message;
                $scope.confirmLabel = options.confirm;
                $scope.rejectLabel = options.reject;

                $scope.showConfirm = !_.isUndefined(options.confirm);
                $scope.showReject = !_.isUndefined(options.reject);

                $scope.confirm = function () {
                    return $xlModalInstance.close();
                };
                $scope.reject = function () {
                    return $xlModalInstance.dismiss('cancel');
                };
            }
        }).result;
    };

    var openConfirmDialog = function openConfirmDialog(options) {
        return _.isUndefined(options.showIf) || options.showIf ? showDialog(options) : $q.resolve();
    };

    return {
        openConfirmDialog: openConfirmDialog
    };
});
'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var CacheSection = function () {
    function CacheSection() {
        _classCallCheck(this, CacheSection);
    }

    _createClass(CacheSection, [{
        key: 'removeAll',
        value: function removeAll() {
            for (var x in this) {
                if (this.hasOwnProperty(x)) {
                    delete this[x];
                }
            }
        }
    }]);

    return CacheSection;
}();

var TabCacheService = function () {
    function TabCacheService() {
        _classCallCheck(this, TabCacheService);
    }

    _createClass(TabCacheService, [{
        key: '$get',
        value: function $get($uiTabs) {
            var cache = new Map();

            var activeTabCache = function activeTabCache() {
                var activeTabId = $uiTabs.getActiveTab().$$id;
                if (cache.has(activeTabId) === false) {
                    cache.set(activeTabId, new Map());
                }
                return cache.get(activeTabId);
            };

            var get = function get(key) {
                var tab = $uiTabs.getActiveTab();
                if (tab) {
                    var value = activeTabCache().get(key);
                    if (!value) {
                        value = new CacheSection();
                        value.$$id = tab.$$id;
                        set(key, value);
                    }
                    return value;
                }
                return undefined;
            };

            var set = function set(key, value) {
                if ($uiTabs.getActiveTab()) {
                    activeTabCache().set(key, value);
                }
            };

            var remove = function remove(key) {
                if ($uiTabs.getActiveTab()) {
                    activeTabCache().delete(key);
                }
            };

            var clear = function clear() {
                if ($uiTabs.getActiveTab()) {
                    activeTabCache().clear();
                }
            };

            var getCurrentTask = function getCurrentTask() {
                var task = get('task');
                if (_.isUndefined(task)) {
                    task = {};
                    set('task', task);
                }
                return task;
            };

            var updateCurrentTask = function updateCurrentTask(newData, $scope) {
                var task = getCurrentTask();
                angular.extend(task, newData);
                if (!_.isUndefined($scope)) {
                    $scope.task = task;
                }
            };

            return {
                get: get,
                set: set,
                remove: remove,
                clear: clear,
                getCurrentTask: getCurrentTask,
                updateCurrentTask: updateCurrentTask
            };
        }
    }]);

    return TabCacheService;
}();

angular.module('xl.online.commons').provider('tabCacheService', TabCacheService);
'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var DeploymentCache = function () {
    function DeploymentCache(tabCacheService) {
        _classCallCheck(this, DeploymentCache);

        this.tabCacheService = tabCacheService;
    }

    _createClass(DeploymentCache, [{
        key: 'mappingScreen',
        get: function get() {
            return this.tabCacheService.get('deployment.mapping');
        }
    }]);

    return DeploymentCache;
}();

angular.module('xl.online.commons').factory('deploymentCache', function (tabCacheService) {
    return new DeploymentCache(tabCacheService);
});
'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var Utils = function () {
    function Utils() {
        _classCallCheck(this, Utils);
    }

    _createClass(Utils, [{
        key: '$get',
        value: function $get() {
            var getBaseName = function getBaseName(id) {
                return _.last(_.split(id, '/'));
            };

            var removePrefix = function removePrefix(id) {
                if (angular.isUndefined(id)) {
                    return id;
                }

                var index = id.indexOf('/');
                if (index === 0) {
                    index = id.indexOf('/', 1);
                    index = index === -1 ? 0 : index;
                }
                return index === -1 ? id : id.substr(index + 1);
            };

            var removeLeadingSlash = function removeLeadingSlash(inputString) {
                return inputString && inputString.replace(/^\//, '');
            };

            var removeTrailingSlash = function removeTrailingSlash(inputString) {
                return inputString && inputString.replace(/\/$/, '');
            };

            var addLeadingSlashIfNotExists = function addLeadingSlashIfNotExists(inputString) {
                if (!_.isUndefined(inputString) && inputString.length !== 0 && !_.startsWith(inputString, '/')) {
                    return '/' + inputString;
                }
                return inputString;
            };

            var getCiIdsWithoutPrefix = function getCiIdsWithoutPrefix(response) {
                return _.map(response.data, function (ci) {
                    return removePrefix(ci.ref);
                });
            };

            return {
                addLeadingSlashIfNotExists: addLeadingSlashIfNotExists,
                getBaseName: getBaseName,
                getCiIdsWithoutPrefix: getCiIdsWithoutPrefix,
                removePrefix: removePrefix,
                removeLeadingSlash: removeLeadingSlash,
                removeTrailingSlash: removeTrailingSlash
            };
        }
    }]);

    return Utils;
}();

angular.module('xl.online.commons').provider('utils', Utils);
'use strict';

angular.module('xl.online.commons').factory('typeSystemService', function ($q, constantService, metadataResource) {
    var typesDescriptors = {};
    var deployableTypes = [];
    var deployableTypesNames = [];
    var containerTypesNames = [];
    var groupedByDeployableType = {};
    var groupedBySuperType = {};

    var loadTypesDescriptors = function loadTypesDescriptors() {
        var deferred = $q.defer();

        if (deployableTypesNames.length > 0) {
            deferred.resolve();
            return deferred.promise;
        }

        metadataResource.listAllDescriptors().then(function (descriptors) {
            deployableTypesNames.length = 0;
            _.forEach(descriptors.data, function (metadata) {
                typesDescriptors[metadata.type] = metadata;

                _.forEach(metadata.superTypes, function (superType) {
                    if (angular.isUndefined(groupedBySuperType[superType])) {
                        groupedBySuperType[superType] = [];
                    }
                    groupedBySuperType[superType].push(metadata);
                });

                if (_.includes(metadata.interfaces, constantService.TYPE.UDM_DEPLOYABLE)) {
                    deployableTypesNames.push(metadata.type);
                    deployableTypes.push(metadata);
                }
                if (_.includes(metadata.interfaces, constantService.TYPE.UDM_CONTAINER)) {
                    containerTypesNames.push(metadata.type);
                }
            });
            groupedByDeployableType = _.groupBy(descriptors.data, 'deployableType');

            // sort deployableTypes
            deployableTypes = _.sortBy(deployableTypes, 'type');
            containerTypesNames = _.sortBy(containerTypesNames);

            deferred.resolve();
        }, function (error) {
            deferred.reject(error);
        });

        return deferred.promise;
    };

    var getTypesByDeployableType = function getTypesByDeployableType(deployableType) {
        return groupedByDeployableType[deployableType];
    };

    var getContainerTypeNames = function getContainerTypeNames() {
        return containerTypesNames;
    };

    var getDeployableTypeNames = function getDeployableTypeNames() {
        return deployableTypesNames;
    };

    var getDeployableTypes = function getDeployableTypes() {
        return deployableTypes;
    };

    var getTypeDescriptorOf = function getTypeDescriptorOf(type) {
        return typesDescriptors[type];
    };

    var getTypeDescriptorOfSuperType = function getTypeDescriptorOfSuperType(superType) {
        return groupedBySuperType[superType];
    };

    var isSourceArtifactType = function isSourceArtifactType(type) {
        return _.includes(typesDescriptors[type].interfaces, 'udm.SourceArtifact');
    };

    return {
        loadTypesDescriptors: loadTypesDescriptors,
        getTypeDescriptorOf: getTypeDescriptorOf,
        getDeployableTypeNames: getDeployableTypeNames,
        getContainerTypeNames: getContainerTypeNames,
        getTypesByDeployableType: getTypesByDeployableType,
        getDeployableTypes: getDeployableTypes,
        getTypeDescriptorOfSuperType: getTypeDescriptorOfSuperType,
        isSourceArtifactType: isSourceArtifactType
    };
});
'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var StatesUtils = function () {
    function StatesUtils() {
        _classCallCheck(this, StatesUtils);

        this.states = {
            UNREGISTERED: 'UNREGISTERED',
            PENDING: 'PENDING',
            SCHEDULED: 'SCHEDULED',
            QUEUED: 'QUEUED',
            EXECUTING: 'EXECUTING',
            FAILING: 'FAILING',
            FAILED: 'FAILED',
            STOPPING: 'STOPPING',
            STOPPED: 'STOPPED',
            ABORTING: 'ABORTING',
            ABORTED: 'ABORTED',
            EXECUTED: 'EXECUTED',
            DONE: 'DONE',
            CANCELLED: 'CANCELLED',
            CANCELLING: 'CANCELLING',
            PAUSED: 'PAUSED',
            SKIP: 'SKIP',
            SKIPPED: 'SKIPPED'
        };
        this.statesByName = _.invert(this.states);
    }

    _createClass(StatesUtils, [{
        key: 'parse',
        value: function parse(state) {
            return this.statesByName[state.toUpperCase()];
        }
    }, {
        key: 'isExecutingSteps',
        value: function isExecutingSteps(state) {
            var parsedState = this.parse(state);
            return parsedState === this.states.QUEUED || parsedState === this.states.EXECUTING || parsedState === this.states.FAILING || parsedState === this.states.STOPPING || parsedState === this.states.ABORTING || parsedState === this.states.CANCELLING;
        }
    }, {
        key: 'isBlockExecutionFinished',
        value: function isBlockExecutionFinished(state) {
            var parsedState = this.parse(state);
            return parsedState === this.states.FAILED || parsedState === this.states.DONE || parsedState === this.states.STOPPED || parsedState === this.states.EXECUTED || parsedState === this.states.DONE || parsedState === this.states.CANCELLED;
        }
    }]);

    return StatesUtils;
}();

angular.module('xl.online.commons').factory('statesUtils', function () {
    return new StatesUtils();
});
'use strict';

angular.module('xl.online.commons').factory('metadataResource', function ($http, xlDeployServicesRoot) {
    return {
        listAllDescriptors: function listAllDescriptors() {
            return $http.get(xlDeployServicesRoot + 'metadata/type');
        },
        listOrchestrators: function listOrchestrators() {
            return $http.get(xlDeployServicesRoot + 'metadata/orchestrators');
        },
        getCandidateValues: function getCandidateValues(propertyName, ci) {
            return $http.post(xlDeployServicesRoot + 'repository/candidate-values?propertyName=' + propertyName + '&resultsPerPage=-1', ci);
        }
    };
});
'use strict';

var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var IdUtils = function () {
    function IdUtils() {
        _classCallCheck(this, IdUtils);
    }

    _createClass(IdUtils, [{
        key: '$get',
        value: function $get() {
            var splitId = function splitId(id) {
                if (_.isArray(id)) {
                    return id;
                } else {
                    return id.split('/');
                }
            };

            return {
                parseInfrastructureDeployedId: function parseInfrastructureDeployedId(id) {
                    var _splitId = splitId(id),
                        _splitId2 = _slicedToArray(_splitId, 3),
                        section = _splitId2[0],
                        host = _splitId2[1],
                        deployable = _splitId2[2];

                    return {
                        section: section,
                        host: host,
                        deployable: deployable
                    };
                },
                parseDeployableId: function parseDeployableId(id) {
                    var _splitId3 = splitId(id),
                        _splitId4 = _slicedToArray(_splitId3, 4),
                        section = _splitId4[0],
                        application = _splitId4[1],
                        version = _splitId4[2],
                        deployable = _splitId4[3];

                    return {
                        section: section,
                        application: application,
                        version: version,
                        deployable: deployable,
                        isPartOf: function isPartOf(packageId) {
                            return _.isEqual(packageId.section, section) && packageId.application === application && packageId.version === version;
                        }
                    };
                },
                parsePackageId: function parsePackageId(id) {
                    var _splitId5 = splitId(id),
                        _splitId6 = _slicedToArray(_splitId5, 3),
                        section = _splitId6[0],
                        application = _splitId6[1],
                        version = _splitId6[2];

                    return {
                        section: section,
                        application: application,
                        version: version
                    };
                }
            };
        }
    }]);

    return IdUtils;
}();

angular.module('xl.online.commons').provider('idUtils', IdUtils);
'use strict';

angular.module('xl.online.commons.utils', []);
'use strict';

angular.module('xl.online.commons.utils').factory('errorService', function () {
    var processDeployment = function processDeployment(data, config) {
        var messages = data.application['validation-messages'];
        if (config && config.type === 'plainText') {
            var msgs = [];
            _.forEach(messages, function (message) {
                msgs.append(message);
            });
            return msgs;
        }
        return messages;
    };

    var processData = function processData(data, config) {
        if (data.id && _.startsWith(data.id, 'deployment-')) {
            return processDeployment(data, config);
        } else {
            return data;
        }
    };

    var isGenericError = function isGenericError(error) {
        return _.isEqual(processData(error), error);
    };

    return {
        isGenericError: isGenericError
    };
});
'use strict';

angular.module('xl.online.commons').factory('constantService', function () {
    return {
        ROOT: {
            APPLICATIONS: 'Applications',
            ENVIRONMENTS: 'Environments',
            INFRASTRUCTURE: 'Infrastructure',
            CONFIGURATION: 'Configuration'
        },
        TYPE: {
            UDM_COMPOSITE_PACKAGE: 'udm.CompositePackage',
            UDM_DEPLOYMENT_PACKAGE: 'udm.DeploymentPackage',
            UDM_APPLICATIONS: 'udm.Application',
            CORE_DIRECTORY: 'core.Directory',
            UDM_ENVIRONMENT: 'udm.Environment',
            UDM_CONTAINER: 'udm.Container',
            UDM_DICTIONARY: 'udm.Dictionary',
            TRIGGER_TRIGGER: 'trigger.Trigger',
            UDM_DEPLOYABLE: 'udm.Deployable',
            UDM_VERSION: 'udm.Version',
            UDM_DEPLOYED_APPLICATION: 'udm.DeployedApplication'
        },
        MODES: {
            EDITION: 'edition',
            VIEW: 'visual',
            CREATION: 'creation'
        }
    };
});
'use strict';

angular.module('xl.online.commons').factory('ciService', function (constantService, ciResource, utils) {

    var ciExists = function ciExists(root, ciName) {
        var ci = utils.removeLeadingSlash(ciName);
        ci = utils.removeTrailingSlash(ci);
        ci = utils.addLeadingSlashIfNotExists(ci);
        return ciResource.ciExists(root + ci);
    };

    var createJsonCi = function createJsonCi(ciObj, type) {
        return { id: ciObj.id, type: type };
    };

    var createJsonCiWithData = function createJsonCiWithData(ciObj, type, data) {
        var jsonCi = createJsonCi(ciObj, type);
        angular.extend(jsonCi, data);
        return jsonCi;
    };

    var extractCiStructure = function extractCiStructure(ciId) {
        if (angular.isUndefined(ciId)) {
            return { root: undefined, name: undefined, path: undefined, id: undefined, idWithoutRoot: undefined };
        }
        var firstSlashIndex = ciId.indexOf('/');
        var root = ciId.substring(0, firstSlashIndex);

        var lastSlashIndex = ciId.lastIndexOf('/');
        var name = ciId.substring(lastSlashIndex + 1);

        var path = ciId.substring(firstSlashIndex, lastSlashIndex);
        path = path.charAt(0) === '/' ? path : '/' + path;
        path = path.length === 1 ? '' : path;

        return { root: root, name: name, path: path, id: root + path + '/' + name, idWithoutRoot: path + '/' + name };
    };

    var normalizeCiId = function normalizeCiId(root, path, name) {
        var parsedRoot = utils.removeTrailingSlash(utils.removeLeadingSlash(root));
        var parsedPath = utils.removeTrailingSlash(utils.addLeadingSlashIfNotExists(path));
        var parsedName = utils.removeTrailingSlash(utils.addLeadingSlashIfNotExists(name));

        var data = {};

        data.name = parsedName ? parsedName : '';
        data.path = parsedPath ? parsedPath : '';
        data.root = parsedRoot ? parsedRoot : '';
        data.id = data.root + data.path + data.name;
        data.idWithoutRoot = data.path + data.name;
        return data;
    };

    var createDirectoriesJsonCis = function createDirectoriesJsonCis(ciObj) {
        var cis = [];
        var folderPrefix = '';

        var path = ciObj.path;

        if (path && path !== '') {
            path = utils.removeLeadingSlash(path);

            var folders = path.split('/');
            for (var i = 0; i < folders.length; i++) {
                var directoryObj = normalizeCiId(ciObj.root, folderPrefix, folders[i]);
                cis.push(createJsonCi(directoryObj, constantService.TYPE.CORE_DIRECTORY));
                folderPrefix += folders[i] + '/';
            }
        }
        return cis;
    };

    var getCi = function getCi(ciId) {
        return ciResource.get(ciId);
    };

    return {
        getCi: getCi,
        ciExists: ciExists,
        createJsonCi: createJsonCi,
        createJsonCiWithData: createJsonCiWithData,
        createDirectoriesJsonCis: createDirectoriesJsonCis,
        normalizeCiId: normalizeCiId,
        extractCiStructure: extractCiStructure
    };
});