import os
import tempfile
from com.xebialabs.deployit.plugin.openshift import LoaderUtil
from java.nio.file import Files, Paths, StandardCopyOption

from kubernetes.client.configuration import ConfigurationObject
from openshift.client import ApiClient


class OpenShiftClientFactory:
    def __init__(self, container):
        self.__server = container

    def create(self):
        if not self.__server.verifyCertificates:
            self.__disable_https_warnings()
        return ApiClient(host=self.__server.serverUrl, config=self.__get_config())

    def __get_config(self):
        config = ConfigurationObject()
        config.verify_ssl = self.__server.verifyCertificates
        config.api_key['authorization'] = "Bearer {0}".format(self.__server.openshiftToken)
        config.ssl_ca_cert = OpenShiftClientFactory.__set_ca_bundle_path()
        return config

    @staticmethod
    def __disable_https_warnings():
        import urllib3
        urllib3.disable_warnings()

    @staticmethod
    def __set_ca_bundle_path():
        ca_bundle_path = OpenShiftClientFactory.__extract_file_from_jar("certifi/cacert.pem")
        os.environ['REQUESTS_CA_BUNDLE'] = ca_bundle_path
        return ca_bundle_path

    @staticmethod
    def __extract_file_from_jar(config_file):
        file_url = LoaderUtil.getResourceBySelfClassLoader(config_file)
        if file_url:
            tmp_file = None
            try:
                tmp_file, tmp_abs_path = tempfile.mkstemp()
                Files.copy(file_url.openStream(), Paths.get(tmp_abs_path), StandardCopyOption.REPLACE_EXISTING)
                return tmp_abs_path
            finally:
                if tmp_file:
                    tmp_file.close()
        else:
            return None
