from xld.kubernetes.pod.pod_helper import PodHelper
from kubernetes import client
from kubernetes.client import ApiClient
from openshift.client import V1DeploymentConfig, V1DeploymentConfigSpec, V1DeploymentStrategy, \
    V1CustomDeploymentStrategyParams, V1RollingDeploymentStrategyParams, V1DeploymentTriggerPolicy, \
    V1DeploymentTriggerImageChangeParams


class DeploymentConfigHelper(object):

    @staticmethod
    def read_deployment_config(deployed):
        for pod_template in deployed.template:
            template = PodHelper.read_pod(PodHelper(), deployed_pod=pod_template, pod=client.V1Pod())
        spec = V1DeploymentConfigSpec(replicas=deployed.replicas, paused=deployed.paused, test=deployed.test,
                                      selector=deployed.selector,
                                      revision_history_limit=deployed.revision_history_limit,
                                      min_ready_seconds=deployed.min_ready_seconds,
                                      strategy=DeploymentConfigHelper.__get_deployment_strategy(deployed.strategy),
                                      template=template,
                                      triggers=DeploymentConfigHelper.__get_deployment_triggers(deployed.triggers))
        deployment_config = V1DeploymentConfig(api_version="v1", kind="DeploymentConfig",
                                               metadata=client.V1ObjectMeta(
                                                   name=DeploymentConfigHelper.get_deployment_config_name(deployed)),
                                               spec=spec)
        return ApiClient().sanitize_for_serialization(deployment_config)

    @staticmethod
    def get_deployment_config_name(deployed):
        return deployed.deploymentConfigName if deployed.deploymentConfigName else deployed.name

    @staticmethod
    def enrich_app_selectors(deployed):
        for pod_template in deployed.template:
            if not pod_template.labels or 'app' not in pod_template.labels:
                pod_template.labels=dict(pod_template.labels, app = DeploymentConfigHelper.get_deployment_config_name(deployed))

    @staticmethod
    def __get_deployment_strategy(strategy_set):

        for strategy in strategy_set:
            if strategy.type == "openshift.CustomDeploymentStrategyParamsSpec":
                return V1DeploymentStrategy(type="Custom", annotations=strategy.annotations,
                                            labels=strategy.labels,
                                            custom_params=V1CustomDeploymentStrategyParams(command=strategy.command,
                                                                                           image=strategy.image,
                                                                                           environment=strategy.environment))
            elif strategy.type == "openshift.RollingDeploymentStrategyParamsSpec":
                return V1DeploymentStrategy(type="Rolling", rolling_params=V1RollingDeploymentStrategyParams(
                    max_surge=strategy.max_surge,
                    max_unavailable=strategy.max_unavailable,
                    timeout_seconds=strategy.timeout_seconds,
                    update_period_seconds=strategy.update_period_seconds,
                    interval_seconds=strategy.interval_seconds,
                    post=DeploymentConfigHelper.__get_deployment_strategy_params(strategy.post)))

            elif strategy.type == "openshift.RecreateDeploymentStrategyParamsSpec":
                return V1DeploymentStrategy(type="Recreate",
                                            recreate_params=V1RollingDeploymentStrategyParams(
                                                timeout_seconds=strategy.timeout_seconds,
                                                pre=DeploymentConfigHelper.__get_deployment_strategy_params(
                                                    strategy.pre),
                                                post=DeploymentConfigHelper.__get_deployment_strategy_params(
                                                    strategy.post)))

            else:
                return None

    @staticmethod
    def __get_deployment_triggers(triggers):
        list_of_triggers = []
        for trigger in triggers:
            list_of_triggers.append(V1DeploymentTriggerPolicy(type=trigger.type_of_trigger,
                                                              image_change_params=V1DeploymentTriggerImageChangeParams(
                                                                  automatic=True if trigger.automatic else False,
                                                                  container_names=trigger.container_names,
                                                                  _from=trigger.from_object,
                                                                  last_triggered_image=trigger.last_triggered_image)))
        return list_of_triggers

    @staticmethod
    def __get_deployment_strategy_params(during):
        return during if during else None
