package com.xebialabs.xldeploy.provisioner.api

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.{RepositoryService, WorkDir}
import com.xebialabs.xldeploy.provisioner._

import scala.collection.convert.wrapAll._
import scala.collection.mutable
import scala.reflect.ClassTag
import scala.util.Try

object CiReferencesSupport {

  implicit class ReferenceResolver[T <: ConfigurationItem](ci: T) {
    def resolved(implicit repositoryService: RepositoryService, workDirectory: WorkDir, resolvedSoFar: mutable.Map[String, ConfigurationItem] = mutable.Map.empty): T = {
      resolvedSoFar.put(ci.getId, ci)
      val descriptors = ci.getType.getDescriptor.getPropertyDescriptors.filter(_.getReferencedType != null)
      descriptors.map(pd => pd -> ci).collect {
        case SetOfCi(pd, values) =>
          pd.set(ci, new JHashSet(values.map(resolveCi)))
        case ListOfCi(pd, values) =>
          pd.set(ci, new JArrayList(values.map(resolveCi)))
        case Ci(pd, refCi) =>
          pd.set(ci, resolveCi(refCi))
      }
      ci
    }

    private def resolveCi(ci: ConfigurationItem)(implicit repositoryService: RepositoryService, workDirectory: WorkDir, resolvedSoFar: mutable.Map[String, ConfigurationItem]): ConfigurationItem = {
      if (resolvedSoFar.contains(ci.getId)) {
        resolvedSoFar.get(ci.getId).get
      } else {
        val resolvedItem = Try(Option(repositoryService.read[ConfigurationItem](ci.getId, workDirectory))).toOption.flatten.getOrElse(ci.resolved)
        resolvedSoFar.putIfAbsent(resolvedItem.getId, resolvedItem)
        resolvedItem
      }
    }
  }


  private val Ci = new PropertyOfType[ConfigurationItem]
  private val ListOfCi = new PropertyOfType[JList[ConfigurationItem]]
  private val SetOfCi = new PropertyOfType[JSet[ConfigurationItem]]

  private class PropertyOfType[T: ClassTag]() {
    lazy val tClass = implicitly[ClassTag[T]].runtimeClass
    def unapply(t: (PropertyDescriptor, ConfigurationItem)): Option[(PropertyDescriptor, T)] = t match {
      case (pd, ci) if Option(pd.get(ci)).exists(value => tClass.isAssignableFrom(value.getClass)) && pd.getReferencedType != null =>
        Option(pd -> pd.get(ci).asInstanceOf[T])
      case _ =>
        None
    }
  }
}
