package com.xebialabs.xldeploy.provisioner.api

import akka.actor.ActorSystem
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.engine.tasker.IEngine
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.{RepositoryService, RepositoryServiceHolder, WorkDirFactory}
import com.xebialabs.deployit.service.deployment.{DeployServicesHolder, DeployedProcessorsFactory, DeployedService, DeploymentService}
import com.xebialabs.deployit.service.validation.Validator
import com.xebialabs.xldeploy.provisioner._
import com.xebialabs.xlplatform.endpoints.{AuthenticatedData, ExtensionRoutes}
import com.xebialabs.xlplatform.scheduler.spring.ServiceHolder
import grizzled.slf4j.Logging
import org.springframework.security.core.Authentication
import org.springframework.security.core.context.SecurityContextHolder
import spray.http.{StatusCode, StatusCodes}
import spray.httpx.SprayJsonSupport
import spray.json.DefaultJsonProtocol
import spray.routing._

import scala.collection.convert.wrapAll._

class ProvisionRoute extends ExtensionRoutes with ProvisioningService with SprayJsonSupport with DefaultJsonProtocol with JsonApiFormat with Logging {

  override lazy val deploymentService: DeploymentService = DeployServicesHolder.getGlobalInstance.getDeploymentService
  override lazy val deployedService: DeployedService = DeployServicesHolder.getGlobalInstance.getDeployedService
  override lazy val repositoryService: RepositoryService = RepositoryServiceHolder.getRepositoryService
  override lazy val workDirFactory: WorkDirFactory = new WorkDirFactory("provisioning")
  override lazy val taskEngine: IEngine = ServiceHolder.getTaskExecutionEngine
  override lazy val validator: Validator = DeployServicesHolder.getGlobalInstance.getValidator
  override lazy val deployedProcessorsFactory: DeployedProcessorsFactory = {
    val f = DeployServicesHolder.getGlobalInstance.getDeployedProcessorsFactory
    f.registerDeployedGenerator(Type.valueOf(classOf[Provisionable]), classOf[CardinalityBasedDeployedGenerator])
    f
  }

  override def route(system: ActorSystem): (AuthenticatedData) => Route = (data) => handleExceptions(customErrorHandler) {
    post {
      deployedProcessorsFactory
      pathPrefix("provisioning") {
        path("initial") {
          entity(as[ProvisioningRequest]) { provisioning =>
            complete(authenticate(data.toAuthentication)(withStatusCode(initialProvisioning(provisioning))))
          }
        } ~ path("task") {
          entity(as[Provision]) { provisioning =>
            complete(authenticate(data.toAuthentication)(withStatusCode(createProvisioningTask(provisioning))))
          }
        } ~
          path("destroy") {
            entity(as[DeprovisioningRequest]) { deprovisioningRequest =>
              complete(authenticate(data.toAuthentication)(withStatusCode(deprovision(deprovisioningRequest))))
            }
          } ~
          path("validate") {
            entity(as[Provision]) { provision =>
              complete(authenticate(data.toAuthentication)(withStatusCode(validate(provision))))
            }
          } ~
          path("preview") {
            entity(as[Provision]) { provision =>
              complete(authenticate(data.toAuthentication)(withStatusCode(preview(provision))))
            }
          } ~
          path("preview" / Segment / IntNumber) { (blockId, stepNr) =>
            entity(as[Provision]) { provision =>
              complete(authenticate(data.toAuthentication)(withStatusCode(preview(provision, blockId, stepNr))))
            }
          } ~
          path("map" / "selected") {
            entity(as[SelectedProvision]) { selectedProvision =>
              complete(authenticate(data.toAuthentication)(withStatusCode(generateSelectedProvisioneds(selectedProvision))))
            }
          } ~
          path("map" / "all") {
            entity(as[Provision]) { provision =>
              complete(authenticate(data.toAuthentication)(withStatusCode(generateAllProvisioneds(provision))))
            }
          }
      }
    }
  }

  private val customErrorHandler = ExceptionHandler {
    case ex: Exception =>
      val msg = "Error: " + ex.getMessage
      error(msg, ex)
      complete(StatusCodes.BadRequest, msg)
  }

  private def authenticate[T](auth: Authentication)(pimpFunction: => T): T = {
    try {
      SecurityContextHolder.getContext.setAuthentication(auth)
      pimpFunction
    } finally {
      SecurityContextHolder.clearContext()
    }
  }

  private def withStatusCode[T <: ErrorAware](call: => T): (StatusCode, T) = {
    val result = call
    statusCode(result) -> result
  }

  private def statusCode(errorAware: ErrorAware): StatusCode = errorAware match {
    case provision: ProvisionedBlueprintAware if hasMissingPlaceholders(provision) => StatusCodes.PreconditionFailed
    case ve: ValidationErrors if ve.validationErrors.isDefined => StatusCodes.PreconditionFailed
    case _ => StatusCodes.Created
  }

  def hasMissingPlaceholders(provision: ProvisionedBlueprintAware): Boolean = {
    provision.provisionedBlueprint.getUnresolvedPlaceholders.values().exists(_.isEmpty)
  }
}
