package com.xebialabs.xldeploy.provisioner
package steps

import com.xebialabs.deployit.plugin.api.flow.{PreviewStep, ExecutionContext, Preview, StepExitCode}
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, Scope, StepMetadata, StepPostConstructContext}
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, Container, Environment}
import com.xebialabs.deployit.repository.ChangeSet
import com.xebialabs.xldeploy.provisioner._
import com.xebialabs.xldeploy.provisioner.resolver.TemplateResolver
import com.xebialabs.xldeploy.provisioner.resolver.TemplateResolver.ResolutionResult

import scala.collection.convert.wrapAll._
import scala.reflect.ClassTag
import scala.util.{Failure, Success}

@StepMetadata(name = "create-all-provisioned-cis")
class CreateAllProvisionedItems extends ProvisionedItemsStep with PreviewStep {

  @transient
  private lazy val templateResolver = TemplateResolver(provisionedBlueprint)

  override def validScope: Scope = Scope.POST_PLAN

  @RulePostConstruct
  def validateAndSetParameters(ctx: StepPostConstructContext) {
    validate(ctx)
    if (description == null || description.isEmpty) {
      description = s"Creating provisioned configuration items"
    }
    order = if (order != null) order else 75
    setParameters(ctx)
  }

  override def getPreview: Preview = {
    val provisionedItems = resolveTemplatesOnPackage() ++ resolvedTemplatesPerProvisioned().values.flatten
    val previewMessages = provisionedItems.map {
      case ResolutionResult(_, Success(id), _) =>
        s"Create or update of $id"
      case ResolutionResult(templateId, Failure(ex), _) =>
        s"FAILED on $templateId with message: ${ex.getMessage}"
    }
    Preview.withContents(s"Create of environment ${provisionedBlueprint.getProvisionedEnvironment.getId}\n" + previewMessages.mkString("\n"))
  }

  override def execute(ctx: ExecutionContext): StepExitCode = {
    implicit val changeSet = new ChangeSet
    implicit val theCtx = ctx
    resolveEnvironmentAndTemplates
    repositoryService.execute(changeSet)
    StepExitCode.SUCCESS
  }

  private def resolveEnvironmentAndTemplates(implicit ctx: ExecutionContext, changeSet: ChangeSet) = {
    val environment = createCi[Environment](provisionedBlueprint.getProvisionedEnvironment.getId)
    val cisOnPackage = resolveTemplatesOnPackage()
    val cisPerProvisioned = resolvedTemplatesPerProvisioned()
    val allResolvedResults = cisOnPackage ++ cisPerProvisioned.values.flatten
    throwErrorOnFailedResolution(allResolvedResults)
    setBoundConfigurationItemsOnProvisioneds(cisPerProvisioned)
    setBoundConfigurationItemsOnBlueprint(cisOnPackage)
    val allConfigurationItems = allResolvedResults.map(_.configurationItem.get)
    environment.setMembers(new JHashSet(filterConfigurationItems[Container](allConfigurationItems)))
    environment.setDictionaries(new JArrayList(filterConfigurationItems[UdmDictionaryType](allConfigurationItems)))
    changeSet.createOrUpdate(environment)
  }

  private def throwErrorOnFailedResolution(resolvedItems: Set[ResolutionResult])(implicit ctx: ExecutionContext) = {
    var errorFound = false
    resolvedItems.filter(_.configurationItem.isFailure).foreach { result =>
      errorFound = true
      ctx.logError(s"Could not resolve ${result.templateId}. Cause: ${result.configurationItem.failed.get.getMessage}")
    }
    if (errorFound) {
      throw new RuntimeException("Not all templates have been resolved")
    }
  }

  private[steps] def resolveTemplatesOnPackage(): Set[ResolutionResult] = {
    templateResolver.resolveBoundTemplate(provisioningPackage)
  }

  private def setBoundConfigurationItemsOnBlueprint(resolvedCIs: Set[ResolutionResult])(implicit changeSet: ChangeSet): Unit = {
    val configurationItems = resolvedCIs.map(_.configurationItem.get)
    changeSet.createOrUpdate(new JArrayList(configurationItems))
    provisionedBlueprint.setBoundConfigurationItems(new JHashSet(configurationItems))
  }

  private def setBoundConfigurationItemsOnProvisioneds(resolvedProvisioneds: Map[ProvisionedType, Set[ResolutionResult]])(implicit changeSet: ChangeSet) = {
    resolvedProvisioneds.foreach { case (p, results) =>
      val configurationItems = results.map(_.configurationItem.get)
      changeSet.createOrUpdate(new JArrayList(configurationItems))
      p.boundConfigurationItems = new JHashSet(configurationItems)
    }
  }

  private[steps] def resolvedTemplatesPerProvisioned(): Map[ProvisionedType, Set[ResolutionResult]] = {
    provisioneds.map(p => p -> resolveTemplatesOnProvisioned(p)).toMap
  }

  private[steps] def resolveTemplatesOnProvisioned(provisioned: ProvisionedType): Set[ResolutionResult] = {
    templateResolver.resolveBoundTemplate(provisioned)
  }

  private[this] def filterConfigurationItems[C <: ConfigurationItem : ClassTag](boundCis: Iterable[ConfigurationItem]): Iterable[C] = {
    boundCis.withFilter(ci => ci.getType.getDescriptor.isAssignableTo(implicitly[ClassTag[C]].runtimeClass)).map(_.asInstanceOf[C])
  }
}
