package com.xebialabs.xldeploy.provisioner
package api

import java.util.UUID

import com.xebialabs.deployit.checks.Checks._
import com.xebialabs.deployit.engine.api.dto.Deployment
import com.xebialabs.deployit.engine.api.dto.Deployment.DeploymentType
import com.xebialabs.deployit.engine.api.execution.{StepState, TaskPreviewBlock}
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, DeployedApplication, Environment}
import com.xebialabs.deployit.repository.{RepositoryService, WorkDir, WorkDirFactory}
import com.xebialabs.deployit.security.permission.{DeployitPermissions, Permission}
import com.xebialabs.deployit.server.api.util.IdGenerator.generateId
import com.xebialabs.deployit.service.deployment._
import com.xebialabs.deployit.service.replacement.{ConsolidatedDictionary, Dictionaries, DictionaryValueException}
import com.xebialabs.deployit.service.validation.Validator
import com.xebialabs.xldeploy.provisioner._
import com.xebialabs.xldeploy.provisioner.api.CiReferencesSupport._
import com.xebialabs.xldeploy.provisioner.resolver.TemplateResolver
import com.xebialabs.xldeploy.provisioner.resolver.placeholder.MissingTemplatePlaceholderCollector
import grizzled.slf4j.Logging

import scala.collection.convert.wrapAll._
import scala.reflect.ClassTag
import scala.util.Success

trait ProvisioningService extends Logging {

  import com.xebialabs.xldeploy.provisioner.api.ValidationSupport._
  import Implicits._

  val deploymentService: DeploymentService
  implicit val repositoryService: RepositoryService
  val deployedService: DeployedService
  val workDirFactory: WorkDirFactory
  val taskEngine: IEngine
  implicit val validator: Validator
  val deployedProcessorsFactory: DeployedProcessorsFactory

  def initialProvisioning(provisioning: ProvisioningRequest): Provision = {
    info(s"Initial provisioning of ${provisioning.packageId} onto ${provisioning.provisioningEnvironmentId}")
    val provisioningPackage = checkPermission[ProvisioningPackage](repositoryService.read(provisioning.packageId))
    val cloudEnvironment = checkPermission[ProvisioningEnvironment](repositoryService.read(provisioning.provisioningEnvironmentId))
    implicit val missingPlaceholdersAggregator = new JHashSet[String]()
    val provisionedBlueprint = createProvisionedBlueprint(provisioning, provisioningPackage, cloudEnvironment)
    findMissingPlaceholdersInProvisioneds(provisionedBlueprint, provisioning.somePlaceholders, provisioningPackage, cloudEnvironment)
    provisionedBlueprintWithMissingPlaceholder(provisionedBlueprint).map(pb => Provision(provisionedBlueprint = pb))
      .getOrElse(validateProvision(
        Provision(provisionedBlueprint = provisionedBlueprint)
      ))
  }

  private def findMissingPlaceholdersInProvisioneds(provisionedBlueprint: ProvisionedBlueprint, userProvidedPlaceholders: Map[String, String], provisioningPackage: ProvisioningPackage, provisioningEnvironment: ProvisioningEnvironment)(implicit missingPlaceholdersAggregator: JHashSet[String]) = {
    val dic = dictionaries(provisioningEnvironment.getDictionaries, userProvidedPlaceholders.filterNot { case (_, value) => value.isEmpty })
    missingPlaceholdersAggregator.addAll(deployedService.generateSelectedDeployeds(provisionedBlueprint, provisioningPackage.getDeployables.toList, provisioningEnvironment.getMembers, dic).getUnresolvedPlaceholders)
  }

  def createProvisioningTask(provision: Provision): ProvisioningTask = {
    info(s"Creating provisioning task for ${provision.provisionedBlueprint.getId}")
    implicit val workDir = workDirFactory.newWorkDir()
    val validatedProvision = validateProvision(resolve(provision))
    if (validatedProvision.validationErrors.isDefined) {
      ProvisioningTask(validationErrors = validatedProvision.validationErrors)
    } else {
      ProvisioningTask(id = Option(createAndRegisterInitialTask(provision)))
    }
  }

  def validate(provision: Provision): Provision = {
    info(s"Validate of ${provision.provisionedBlueprint.getId}")
    implicit val workDir = workDirFactory.newWorkDir()
    validateProvision(resolve(provision))
  }

  def deprovision(deprovisioningRequest: DeprovisioningRequest): ProvisioningTask = {
    info(s"De-provisioning of ${deprovisioningRequest.provisionedBlueprintId}")
    implicit val workDir = workDirFactory.newWorkDir()
    val provisionedBlueprint = checkPermission[ProvisionedBlueprint](repositoryService.read(deprovisioningRequest.provisionedBlueprintId))
    createDeprovisionSpecification(provisionedBlueprint)
  }

  def preview(provision: Provision): TaskPreview = {
    info(s"Preview of ${provision.provisionedBlueprint.getId}")
    implicit val workDir = workDirFactory.newWorkDir()
    val validated = validateProvision(provision)
    val someValidationErrors: Option[Seq[ValidationError]] = validated.validationErrors
    if (someValidationErrors.isDefined) {
      TaskPreview(validationErrors = someValidationErrors)
    } else {
      val taskSpec = createTaskSpecification(provision)
      TaskPreview(preview = Some(new TaskPreviewBlock(provision.provisionedBlueprint.getId, taskSpec.getBlock)))
    }
  }

  def preview(provision: Provision, blockId: String, stepNr: Int): StepPreview = {
    val taskPreview: TaskPreview = preview(provision)
    taskPreview.preview match {
      case Some(p) =>
        val topBlock = p.getBlock.asInstanceOf[Block]
        val childBlock = topBlock.getBlock(BlockPath.apply(blockId).tail)
        childBlock match {
          case Some(child: StepBlock) =>
            val steps = child.getSteps()
            if (stepNr > 0 && stepNr <= steps.length) {
              val stepState: StepState = steps.get(stepNr - 1)
              if (hasPermission(DeployitPermissions.TASK_PREVIEWSTEP)) {
                StepPreview(preview = Some(new PreviewStepAdapter(stepState.asInstanceOf[TaskStep])))
              } else {
                StepPreview(preview = Some(stepState))
              }
            } else {
              StepPreview(validationErrors = Some(Seq(ValidationError("", None, message = s"Not a valid step number [$stepNr]"))))
            }
          case _ => StepPreview(validationErrors = Some(Seq(ValidationError("", None, message = s"Block [$blockId}] is a composite block so it has no steps"))))
        }
      case _ => StepPreview(validationErrors = taskPreview.validationErrors)
    }
  }

  def generateAllProvisioneds(provision: Provision): Provision = {
    implicit val missingPlaceholdersAggregator = new JHashSet[String]()
    val provisionedBlueprint: ProvisionedBlueprint = provision.provisionedBlueprint
    val provisioningPackage = checkPermission[ProvisioningPackage](repositoryService.read(provisionedBlueprint.getVersion.getId))
    val provisioningEnvironment = checkPermission[ProvisioningEnvironment](repositoryService.read(provisionedBlueprint.getProvisioningEnvironment.getId))
    val allProvisionableIds = provisioningPackage.getDeployables.map(_.getId).toSeq
    val provisioneds = generateProvisioneds(provisionedBlueprint, allProvisionableIds, provisioningPackage, provisioningEnvironment, provisionedBlueprint.getUnresolvedPlaceholders.toMap)
    val maybeEnvironment = createProvisionedEnvironment(provisioningEnvironment, provisionedBlueprint)
    provisionedBlueprintWithMissingPlaceholder(provisionedBlueprint).map(bp => Provision(provisionedBlueprint = bp))
      .getOrElse(validateProvision(Provision(provisionedBlueprint = provisionedBlueprint.withEnvironment(maybeEnvironment), provisioneds = provisioneds.toOption)))
  }

  def generateSelectedProvisioneds(selectedProvision: SelectedProvision): Provision = {
    implicit val missingPlaceholdersAggregator = new JHashSet[String]()
    val provisionableIds = selectedProvision.selectedProvisionableIds
    val provision = selectedProvision.provision
    val provisionedBlueprint: ProvisionedBlueprint = provision.provisionedBlueprint
    val provisioningPackage = checkPermission[ProvisioningPackage](repositoryService.read(provisionedBlueprint.getVersion.getId))
    val provisioningEnvironment = checkPermission[ProvisioningEnvironment](repositoryService.read(provisionedBlueprint.getProvisioningEnvironment.getId))
    val provisioneds = generateProvisioneds(provisionedBlueprint, provisionableIds, provisioningPackage, provisioningEnvironment, provisionedBlueprint.getUnresolvedPlaceholders.toMap)
    val maybeEnvironment = createProvisionedEnvironment(provisioningEnvironment, provisionedBlueprint)
    provisionedBlueprintWithMissingPlaceholder(provisionedBlueprint).map(bp => Provision(provisionedBlueprint = bp))
      .getOrElse(validateProvision(Provision(provisionedBlueprint = provisionedBlueprint.withEnvironment(maybeEnvironment), provisioneds = (provisioneds ++ provision.provisioneds.getOrElse(Nil)).toOption)))
  }

  private def generateProvisioneds(provisionedBlueprint: ProvisionedBlueprint, ids: Seq[String], provisioningPackage: ProvisioningPackage, provisioningEnvironment: ProvisioningEnvironment, unresolvedPlaceholders: Map[String, String])(implicit missingPlaceholdersAggregator: JSet[String]): Seq[DeployedType] = {
    val provisionablePerId = provisioningPackage.getDeployables.map(d => d.getId -> d).toMap
    val requiredProvisionables = ids.map(id => provisionablePerId.getOrElse(id, throw new RuntimeException(s"$id is not found in provisioning package ${provisioningPackage.getId}")))
    val dic = Dictionaries.of(provisioningEnvironment.getDictionaries).withAdditionalEntries(unresolvedPlaceholders)
    val generatedDeployeds = deployedService.generateSelectedDeployeds(provisionedBlueprint, requiredProvisionables, provisioningEnvironment.getMembers, dic)
    missingPlaceholdersAggregator.addAll(generatedDeployeds.getUnresolvedPlaceholders)
    generatedDeployeds.getDeployeds
  }

  private def provisionedBlueprintWithMissingPlaceholder(provisionedBlueprint: ProvisionedBlueprint)(implicit missingPlaceholdersAggregator: JHashSet[String]): Option[ProvisionedBlueprint] = {
    if (missingPlaceholdersAggregator.isEmpty) {
      None
    } else {
      val userProvidedPlaceholders: JMap[String, String] = new JHashMap[String, String]()
      userProvidedPlaceholders.putAll(provisionedBlueprint.getUnresolvedPlaceholders)
      missingPlaceholdersAggregator.map(placeholder => userProvidedPlaceholders.put(placeholder, ""))
      provisionedBlueprint.setUnresolvedPlaceholders(userProvidedPlaceholders)
      Option(provisionedBlueprint)
    }
  }

  private def createProvisionedBlueprint(provisioning: ProvisioningRequest, provisioningPackage: ProvisioningPackage, provisioningEnvironment: ProvisioningEnvironment)(implicit missingPlaceholdersAggregator: JSet[String]) = {
    val consolidatedDictionary = dictionaries(provisioningEnvironment.getDictionaries, provisioning.somePlaceholders).consolidate()
    addMissingTemplatePlaceholders(provisioningPackage, consolidatedDictionary)
    val provisionedBlueprint = deployedService.generateDeployedApplication(Type.valueOf(classOf[ProvisionedBlueprint]), provisioningPackage,
      provisioningEnvironment, missingPlaceholdersAggregator, provisioning.somePlaceholders).asInstanceOf[ProvisionedBlueprint]
    val provisioningId = hashId(provisionedBlueprint.getName)
    provisionedBlueprint.setId(generateId(provisioningEnvironment, provisioningId + "-" + provisioningPackage.getDistribution.getName))
    provisionedBlueprint.setProvisioningId(provisioningId)
    provisionedBlueprint.setUnresolvedPlaceholders(provisioning.somePlaceholders)
    provisionedBlueprint
  }

  private def addMissingTemplatePlaceholders(provisioningPackage: ProvisioningPackage, consolidatedDictionary: ConsolidatedDictionary)(implicit missingPlaceholdersAggregator: JSet[String]): Unit = {
    val missingPlaceholderCollector = new MissingTemplatePlaceholderCollector(consolidatedDictionary)
    provisioningPackage.boundTemplates.foreach(t => missingPlaceholdersAggregator.addAll(missingPlaceholderCollector.missingPlaceholders(t)))
  }

  private def createProvisionedEnvironment(provisioningEnvironment: ProvisioningEnvironment, provisionedBlueprint: ProvisionedBlueprint)(implicit missingPlaceholdersAggregator: JSet[String]): Option[Environment] = {
    if (provisionedBlueprint.getEnvironmentName == null) {
      None
    } else {
      val dic = Dictionaries.of(provisioningEnvironment.getDictionaries).withAdditionalEntries(provisionedBlueprint.getUnresolvedPlaceholdersWithValues)
      TemplateResolver(dic.consolidate(), provisionedBlueprint).resolveEnvironmentId()
        .map(createCi[Environment]).map(Option.apply)
        .recoverWith {
          case ex: DictionaryValueException =>
            missingPlaceholdersAggregator.addAll(ex.getMissingValues)
            Success(None)
        }.get
    }
  }

  private def resolve(provision: Provision)(implicit workDir: WorkDir): Provision = {
    val resolvedProvisioned = provision.provisioneds.map(_.map(_.resolved))
    val resolvedProvisionedBlueprint = provision.provisionedBlueprint.resolved
    provision.copy(provisioneds = resolvedProvisioned, provisionedBlueprint = resolvedProvisionedBlueprint)
  }

  private def validateProvision(provision: Provision): Provision = {
    provision.copy(validationErrors = (provision.provisioneds.map(_.validated).getOrElse(Nil) ++ validateBlueprintAware(provision)).toOption)
  }

  private def validateBlueprintAware(aware: ProvisionedBlueprintAware) = {
    aware.provisionedBlueprint.validateWithoutProperty("provisioneds")
  }

  private def createAndRegisterInitialTask(provision: Provision)(implicit workDir: WorkDir): String = {
    val taskSpecification: TaskSpecification = createTaskSpecification(provision)
    taskEngine.register(taskSpecification)
  }

  private def createTaskSpecification(provision: Provision)(implicit workDir: WorkDir): TaskSpecification = {
    checkPermission[ProvisioningPackage](repositoryService.read(provision.provisionedBlueprint.getProvisioningPackage.getId))
    checkPermission[ProvisioningEnvironment](repositoryService.read(provision.provisionedBlueprint.getProvisioningEnvironment.getId))
    val deltaSpec = deploymentService.prepareInitialSpecification(createInitialDeployment(provision), Map.empty[String, DeployedApplication])
    deploymentService.getTaskFullSpecification(deltaSpec, workDir, workDir)
  }

  private def createInitialDeployment(provision: Provision)(implicit workDir: WorkDir): Deployment = {
    import com.xebialabs.xldeploy.provisioner.api.CiReferencesSupport._
    val deployment = new Deployment()
    val maybeConfigurationItems = provision.provisioneds.map(_.map(_.resolved))
    deployment.setDeployeds(new JArrayList(maybeConfigurationItems.getOrElse(Nil)))
    deployment.setDeployedApplication(provision.provisionedBlueprint.resolved)
    deployment.setId("deployment-" + UUID.randomUUID.toString)
    deployment.setDeploymentType(DeploymentType.INITIAL)
    deployment
  }

  private def createDeprovisionSpecification(provisionedBlueprint: ProvisionedBlueprint)(implicit workDir: WorkDir): ProvisioningTask = {
    val deltaSpec = deploymentService.prepareUndeployment(provisionedBlueprint)
    val taskSpecification = deploymentService.getTaskFullSpecification(deltaSpec, workDir)
    val taskId = taskEngine.register(taskSpecification)
    ProvisioningTask(id = Option(taskId))
  }

  private def dictionaries(dictionaries: Seq[Dictionary], userProvidedPlaceholders: Map[String, String]): Dictionaries = {
    Dictionaries.of(dictionaries).withAdditionalEntries(userProvidedPlaceholders)
  }

  private def checkPermission[T <: ConfigurationItem : ClassTag : PermissionChecker](ci: ConfigurationItem): T = {
    val clazz: Class[_] = implicitly[ClassTag[T]].runtimeClass
    val permissionChecker = implicitly[PermissionChecker[T]]
    checkArgument(clazz.isAssignableFrom(ci.getClass), "%s is not a %s", ci, clazz.getSimpleName)
    val typedCi = ci.asInstanceOf[T]
    permissionChecker.checkPermission(typedCi)
    typedCi
  }

  private def hasPermission(permission: Permission): Boolean = permission.getPermissionHandler.hasPermission(null)
}


