package com.xebialabs.xldeploy.provisioner.api

import com.xebialabs.deployit.core.{EncryptedStringValue, StringValue}
import com.xebialabs.deployit.plugin.api.udm.{Container, Deployable}
import com.xebialabs.deployit.server.api.util.IdGenerator
import com.xebialabs.deployit.service.deployment.{DeployedGenerator, DeploymentContext, GeneratedDeployeds}
import com.xebialabs.deployit.service.replacement.{ConsolidatedDictionary, DictionaryValueException}
import com.xebialabs.xldeploy.provisioner._
import com.xebialabs.xldeploy.provisioner.resolver.placeholder.MissingTemplatePlaceholderCollector

import scala.collection.convert.wrapAll._
import scala.util.{Failure, Success, Try}

class CardinalityBasedDeployedGenerator(nextGenerator: DeployedGenerator) extends DeployedGenerator {

  private val cardinalityFieldName = "cardinality"
  private val ordinalFieldName = "ordinal"

  override def generateDeployed(deploymentContext: DeploymentContext, deployable: Deployable, target: Container): GeneratedDeployeds = {
    val deployeds = new GeneratedDeployeds()
    val provisionable = deployable.asInstanceOf[Provisionable]
    val provider = target.asInstanceOf[Provider]
    deployeds.addUnresolvedPlaceholder(findMissingPlaceholdersInTemplates(provisionable.boundTemplates ++ provisionable.provisioners.map(_.hostTemplate), deploymentContext.dictionaryFilteredByContainer(target)))
    deployeds.merge(resolveCardinalityAndCreateOrdinalInstances(deploymentContext, provisionable, provider))
    deployeds
  }

  private def findMissingPlaceholdersInTemplates(templates: JSet[Template], consolidatedDictionary: ConsolidatedDictionary): Set[String] = {
    val missingTemplatePlaceholderCollector = new MissingTemplatePlaceholderCollector(consolidatedDictionary)
    val missingPlaceholders = templates.flatMap(missingTemplatePlaceholderCollector.missingPlaceholders)
    val children = templates.flatMap(_.childTemplates.toSet)
    val placeholderOnChildren = if(children.nonEmpty) {
      findMissingPlaceholdersInTemplates(children, consolidatedDictionary)
    } else {
      Set.empty[String]
    }
    placeholderOnChildren ++ missingPlaceholders
  }

  private def resolveCardinalityAndCreateOrdinalInstances(deploymentContext: DeploymentContext, provisionable: Provisionable, provider: Provider): GeneratedDeployeds = {
    resolveCardinality(provisionable, deploymentContext.dictionaryFilteredByContainer(provider)).map(c => createOrdinalInstances(c, deploymentContext, provisionable, provider)).recover {
      case ex: DictionaryValueException =>
        val generatedDeployeds = nextGenerator.generateDeployed(deploymentContext, provisionable, provider)
        generatedDeployeds.addUnresolvedPlaceholder(ex.getMissingValues)
        generatedDeployeds
      case nfe: NumberFormatException => throw new IllegalArgumentException("cardinality must be resolved to integer", nfe)
    }.get
  }

  private def resolveCardinality(provisionable: Provisionable, consolidatedDictionary: ConsolidatedDictionary): Try[Int] = {
    Try(consolidatedDictionary.resolve(provisionable.cardinality, provisionable.getType.getDescriptor.getPropertyDescriptor(cardinalityFieldName))) match {
      case Success(value: EncryptedStringValue) =>
        Failure(new scala.IllegalArgumentException("Currently, we do not support encrypted values for cardinality"))
      case Success(value: StringValue) =>
        Try(value.toPublicFacingValue.toInt)
      case Failure(e) => Failure(e)
    }
  }

  private def createOrdinalInstances(cardinality: Int, deploymentContext: DeploymentContext, deployable: Provisionable, provider: Provider): GeneratedDeployeds = {
    val provisionedBlueprint = deploymentContext.getAppliedDistribution.asInstanceOf[ProvisionedBlueprint]
    val allGeneratedDeployeds = new GeneratedDeployeds()
    1 to cardinality foreach { case ordinal =>
      val generatedDeployeds = nextGenerator.generateDeployed(deploymentContext.withIdGenerator(new CardinalityIdGenerator(provisionedBlueprint, ordinal)), deployable, provider)
      generatedDeployeds.getDeployeds.foreach { d =>
        d.setProperty(ordinalFieldName, ordinal)
      }
      allGeneratedDeployeds.merge(generatedDeployeds)
    }
    allGeneratedDeployeds
  }

  class CardinalityIdGenerator(provisionedBlueprint: ProvisionedBlueprint, ordinal: Int) extends DeploymentContext.DeployedIdGenerator {
    override def generatedId(container: Container, deployable: Deployable): String = ordinal match {
      case 1 =>
        IdGenerator.generateId(container, provisionedBlueprint.getProvisioningId + "-" + deployable.getName)
      case ord =>
        IdGenerator.generateId(container, provisionedBlueprint.getProvisioningId + "-" + deployable.getName + "-" + ord)
    }
  }
}
