package com.xebialabs.xldeploy.provisioner.steps

import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, StepExitCode}
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, Scope, StepMetadata, StepPostConstructContext}
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot
import com.xebialabs.xldeploy.provisioner.resolver.TemplateResolver
import com.xebialabs.xldeploy.provisioner.resolver.TemplateResolver.ResolutionResult

import scala.util.{Failure, Success, Try}

@StepMetadata(name = "validate-provisioned-cis")
class ValidationStep extends ProvisionedItemsStep {

  override def validScope: Scope = Scope.PRE_PLAN

  @transient
  private[steps] lazy val templateResolver = TemplateResolver(provisionedBlueprint)

  @RulePostConstruct
  def validateAndSetParameters(ctx: StepPostConstructContext) {
    validate(ctx)
    if (description == null || description.isEmpty) {
      description = "Validating provisioning configuration items"
    }
    order = if (order != null) order else 10
    setParameters(ctx)
  }

  override def execute(ctx: ExecutionContext): StepExitCode = {
    val resolvedCIs = templateResolver.resolveBoundTemplate(provisioningPackage).toList ++ provisioneds.flatMap(p => templateResolver.resolveBoundTemplate(p).toList)
    val resolvedCiIds = Try(provisionedBlueprint.getProvisionedEnvironment.getId) +: resolvedCIs.map(_.instanceId)
    val failed = validateBoundConfigurationItems(ctx, resolvedCIs) ||
      validateDirectories(ctx, resolvedCIs) ||
      validateEnvironment(ctx) ||
      validateDuplicateIds(ctx, resolvedCiIds) ||
      validateInvalidChars(ctx, resolvedCiIds) ||
      validateEnvironmentName(ctx)
    if (failed) {
      StepExitCode.FAIL
    } else {
      StepExitCode.SUCCESS
    }
  }

  private[steps] def validateDirectories(ctx: ExecutionContext, resolvedCis: List[ResolutionResult]): Boolean = {
    if (provisionedBlueprint.getDirectoryPath != null) {
      val allRoots = resolvedCis.map(_.instanceId.map(_.split("/").head)).filter(_.isSuccess).map(_.get).::(ConfigurationItemRoot.ENVIRONMENTS.getRootNodeName)
      !allRoots.forall { root =>
        val dirPath = s"$root/${provisionedBlueprint.getDirectoryPath}"
        val exists = repositoryService.exists(dirPath)
        if (!exists) {
          ctx.logError(s"$dirPath does not exist")
        } else {
          ctx.logOutput(s"$dirPath exists")
        }
        exists
      }
    } else {
      false
    }
  }

  private[steps] def validateEnvironmentName(ctx: ExecutionContext): Boolean = {
    val invalidChar = findInvalidChar("/%:[]*|\t\r\n")(provisionedBlueprint.getEnvironmentName)
    if(invalidChar.isDefined) {
      ctx.logError(s"Environment name cannot contain ${invalidChar.get}")
      true
    } else {
      false
    }
  }

  private[steps] def validateEnvironment(ctx: ExecutionContext): Boolean = {
    val id = provisionedBlueprint.getProvisionedEnvironment.getId
    val failed = repositoryService.exists(id)
    if (failed) {
      ctx.logError(s"Environment $id already exists")
    }
    failed
  }

  private[steps] def validateBoundConfigurationItems(ctx: ExecutionContext, resolvedCis: List[ResolutionResult]): Boolean = {
    val idsWithExists = resolvedCis.map(r => r.templateId -> r.instanceId.map(id => id -> repositoryService.exists(id)))
    var failed = false
    idsWithExists.foreach {
      case (_, Success((id, true))) =>
        ctx.logError(s"Configuration item with id: $id already exists in the repository")
        failed = true
      case (templateId, Failure(ex)) =>
        ctx.logError(s"Skipping validation of template $templateId as it was not resolved: ${ex.getMessage}")
      case (_, Success((id, false))) =>
    }
    failed
  }

  private[steps] def validateDuplicateIds(ctx: ExecutionContext, resolvedCis: List[Try[String]]): Boolean = {
    val by = resolvedCis.filter(_.isSuccess).map(_.get).groupBy(id => id)
    val duplicateIds = by.filter(pair => pair._2.length > 1)
    var failed = false
    duplicateIds.foreach {
      case (id, ids) =>
      ctx.logError(s"$id is duplicated")
      failed = true
    }
    failed
  }

  private[steps] def validateInvalidChars(ctx: ExecutionContext, resolvedCis: List[Try[String]]): Boolean = {
    val invalidChars = "%:[]*|\t\r\n"
    var failed = false

    val findInvalid = findInvalidChar(invalidChars)(_)

    def charToString(char: Char) = {
      if(char.isWhitespace) {
        "whitespace"
      } else {
        char.toString
      }
    }

    resolvedCis.filter(_.isSuccess).map(_.get).foreach { id =>
      val invalidChar = findInvalid(id)
      if (invalidChar.isDefined) {
        ctx.logError(s"$id contains illegal character ${charToString(invalidChar.get)}")
        failed = true
      }
    }
    failed
  }

  private[steps] def findInvalidChar(invalidChars: String)(id: String): Option[Char] = {
    invalidChars.find(id.toCharArray.contains)
  }
}
