package com.xebialabs.xldeploy.provisioner
package resolver

import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot
import com.xebialabs.deployit.service.replacement.{ConsolidatedDictionary, Dictionaries}
import com.xebialabs.xldeploy.provisioner.resolver.placeholder.PlaceholderResolver

import scala.collection.convert.wrapAll._
import scala.collection.mutable
import scala.util.Try

object TemplateResolver {

  def apply(provisionedBlueprint: ProvisionedBlueprint): TemplateResolver = apply(Dictionaries.of(provisionedBlueprint.getProvisioningEnvironment.getDictionaries)
    .withAdditionalEntries(provisionedBlueprint.getUnresolvedPlaceholdersWithValues).consolidate(), provisionedBlueprint)

  def apply(dictionary: ConsolidatedDictionary, provisionedBlueprint: ProvisionedBlueprint): TemplateResolver = new TemplateResolver(dictionary, provisionedBlueprint)

  private[resolver] type ResolvedCache = mutable.Map[String, ConfigurationItem]

  private[resolver] type ParentCiId = String

  private[resolver] type TemplateName = String

  private[resolver] def emptyCache = new mutable.HashMap[String, ConfigurationItem]()

  type CiIdGenerator = (ParentCiId, TemplateName) => String

  case class ResolutionResult(templateId: String, instanceId: Try[String], configurationItem: Try[ConfigurationItem])

}

class TemplateResolver(dictionary: ConsolidatedDictionary, provisionedBlueprint: ProvisionedBlueprint) {

  import TemplateResolver._

  private lazy val idGenerator: TemplateResolver.CiIdGenerator = (parent, name) =>
    s"$parent/${provisionedBlueprint.getProvisioningId}-$name"

  private lazy val idGeneratorWithOrdinal: TemplateResolver.CiIdGenerator = (parent, name) =>
    s"$parent/${provisionedBlueprint.getProvisioningId}-$name-{{%ordinal%}}"

  private lazy val simpleIdGenerator: CiIdGenerator = (parent, name) => s"$parent/$name"

  private def pickCorrectIdGenerator(provisioned: Option[ProvisionedType] = None): CiIdGenerator = if (provisioned.exists(_.ordinal > 1))
    idGeneratorWithOrdinal
  else
    idGenerator

  def resolveEnvironmentId(): Try[String] = {
    implicit val resolver = placeholderResolver()
    Try(resolveValue(rootDirectory("Environments") + "/" + provisionedBlueprint.getEnvironmentName + "-" + provisionedBlueprint.getProvisioningId))
  }

  private def placeholderResolver(provisioned: Option[ProvisionedType] = None): PlaceholderResolver = PlaceholderResolver(dictionary, provisioned)

  def resolveBoundTemplate(provisioned: ProvisionedType): Set[ResolutionResult] = {
    val maybeProvisioned = Option(provisioned)
    implicit val resolver = placeholderResolver(maybeProvisioned)
    resolveRootTemplates(provisioned.getDeployable.boundTemplates.toSet, pickCorrectIdGenerator(maybeProvisioned))
  }

  def resolveBoundTemplate(provisioningPackage: ProvisioningPackage): Set[ResolutionResult] = {
    implicit val resolver = placeholderResolver()
    resolveRootTemplates(provisioningPackage.boundTemplates.toSet, pickCorrectIdGenerator())
  }

  def resolveSingleRootTemplate(template: Template, provisioned: ProvisionedType): ResolutionResult = {
    val maybeProvisioned = Option(provisioned)
    implicit val alreadyResolvedTemplates = emptyCache
    implicit val resolver = placeholderResolver(maybeProvisioned)
    resolveTemplateToCi(template, rootDirectory(template.getType.getDescriptor.getRootName), pickCorrectIdGenerator(maybeProvisioned))
  }

  private def resolveRootTemplates(templates: Set[Template], idGenerator: CiIdGenerator)(implicit resolver: PlaceholderResolver): Set[ResolutionResult] = {
    implicit val alreadyResolvedTemplates = emptyCache
    val configurationItems = templates.flatMap(resolveRootTemplate(_, idGenerator))
    templates.foreach(setResolvedReferences)
    configurationItems
  }

  private def resolveRootTemplate(template: Template, idGenerator: CiIdGenerator)(implicit alreadyResolvedTemplates: ResolvedCache, resolver: PlaceholderResolver): Set[ResolutionResult] = {
    val instanceTypeDescriptor = instanceType(template).getDescriptor
    if (instanceTypeDescriptor.getRoot != ConfigurationItemRoot.NESTED) {
      val resolvedConfigurationItems = resolveTemplateHierarchy(template, rootDirectory(instanceTypeDescriptor.getRootName), idGenerator)
      resolvedConfigurationItems
    } else {
      throw new IllegalArgumentException("Cannot initialize a configuration item with a nested root")
    }
  }

  private def rootDirectory(rootName: String) = {
    val pathWithRoot = rootName + "/" + Option(provisionedBlueprint.getDirectoryPath).getOrElse("")
    pathWithRoot.split("/").filterNot(_.isEmpty).mkString("/")
  }

  private def resolveTemplateHierarchy(template: Template, parentId: ParentCiId, idGenerator: CiIdGenerator)(implicit alreadyResolvedTemplates: ResolvedCache, resolver: PlaceholderResolver): Set[ResolutionResult] = {
    val resolutionResult = resolveTemplateToCi(template, parentId, idGenerator)
    resolveTemplateChildren(template, resolutionResult.instanceId, simpleIdGenerator) + resolutionResult
  }

  private def resolveTemplateChildren(template: Template, instanceId: Try[String], idGenerator: CiIdGenerator)(implicit alreadyResolvedTemplates: ResolvedCache, resolver: PlaceholderResolver): Set[ResolutionResult] = {
    if (instanceId.isSuccess) {
      template.childTemplates.flatMap(template => resolveTemplateHierarchy(template, instanceId.get, idGenerator)).toSet
    } else {
      Set()
    }
  }

  private def resolveTemplateToCi(template: Template, parentId: String, idGenerator: CiIdGenerator)(implicit alreadyResolvedTemplates: ResolvedCache, resolver: PlaceholderResolver) = ResolutionResult(template.getId,
    Try(resolveValue(generateInstanceId(template, parentId, idGenerator))),
    Try {
      val instanceId = resolveValue(generateInstanceId(template, parentId, idGenerator))
      val ciType = instanceType(template)
      val ci: ConfigurationItem = createCi(ciType, instanceId)

      template.getType.getDescriptor.getPropertyDescriptors.collect {
        case prop if prop.getReferencedType == null && containsPropertyDescriptor(ci, prop) =>
          val value = template.getProperty[AnyRef](prop.getName)
          if (value.nonEmpty) {
            ci.setProperty(prop.getName, resolver.resolve(value, Option(prop.getKind)))
          }
      }
      alreadyResolvedTemplates.put(template.getId, ci)
      ci
    }
  )

  private def generateInstanceId(template: Template, parentCiId: ParentCiId, idGenerator: CiIdGenerator): String = {
    if (Option(template.instanceName).exists(_.nonEmpty)) {
      simpleIdGenerator(parentCiId, template.instanceName)
    } else {
      idGenerator(parentCiId, template.getName)
    }
  }

  private def setResolvedReferences(template: Template)(implicit alreadyResolvedTemplates: ResolvedCache, resolver: PlaceholderResolver): Unit = {
    val propertyDescriptors = template.getType.getDescriptor.getPropertyDescriptors.toSeq
    alreadyResolvedTemplates.get(template.getId).foreach { resolvedCi =>
      propertyDescriptors.collect {
        case prop if prop.getReferencedType != null && prop.get(template).nonEmpty && containsPropertyDescriptor(resolvedCi, prop) =>
          resolvedCi.setProperty(prop.getName, convertReferenceValue(template, prop))
      }
      template.childTemplates.foreach(setResolvedReferences)
    }
  }

  private def convertReferenceValue(template: Template, propertyDescriptor: PropertyDescriptor)(implicit alreadyResolvedTemplates: ResolvedCache): AnyRef = {
    propertyDescriptor.getKind match {
      case PropertyKind.CI =>
        val parentTemplate = propertyDescriptor.get(template).asInstanceOf[Template]
        findAllReferences(Seq(parentTemplate)).head
      case PropertyKind.LIST_OF_CI if isTemplateReference(propertyDescriptor) =>
        new JArrayList(findAllReferences(propertyDescriptor.get(template).asInstanceOf[JList[Template]]))
      case PropertyKind.SET_OF_CI if isTemplateReference(propertyDescriptor) =>
        new JHashSet(findAllReferences(propertyDescriptor.get(template).asInstanceOf[JSet[Template]]))
      case PropertyKind.LIST_OF_CI | PropertyKind.SET_OF_CI =>
        propertyDescriptor.get(template)
    }
  }

  private def findAllReferences(templates: Iterable[Template])(implicit alreadyResolvedTemplates: ResolvedCache): Iterable[ConfigurationItem] = {
    templates.map(t => alreadyResolvedTemplates.getOrElse(t.getId, throw new IllegalStateException(s"Template with id ${t.getId} does not have a resolved configuration item")))
  }

  private def containsPropertyDescriptor(ci: ConfigurationItem, propertyDescriptor: PropertyDescriptor): Boolean = {
    ci.getType.getDescriptor.getPropertyDescriptor(propertyDescriptor.getName) != null
  }

  private def isTemplateReference(templatePropertyDescriptor: PropertyDescriptor) = {
    templatePropertyDescriptor.getReferencedType.getDescriptor.isAssignableTo(typeOf[Template])
  }

  private def instanceType(template: Template) = {
    typeOf(template.getType.toString.substring(templatePrefix.length))
  }

  def resolveValue(property: String)(implicit resolver: PlaceholderResolver = placeholderResolver()): String = {
    resolver.resolve(property).asInstanceOf[String]
  }
}
