import base64
from terraform.mapper.resource_mapper import ResourceMapper

class GKEClusterMapper(ResourceMapper):
    def __init__(self):
        super(GKEClusterMapper, self).__init__(["k8s.Master", "k8s.Namespace"])
        self.gke_attributes = {}

    def create_ci(self, tf_resource, folder):
        if self.types_supported(tf_resource['type']):
            cluster_name = tf_resource['primary']['id']
            print("Creating CI of type 'k8s.Master' from '%s' cluster" % cluster_name)
            self.gke_attributes = tf_resource['primary']['attributes']
            cluster_id = folder[:folder.rfind('/')+1] + cluster_name + "-GKECluster"
            ca_cert = self.get_attribute('master_auth.0.cluster_ca_certificate', True)
            properties = {'isEKS':         False,
                          'skipTLS':       ca_cert is None,
                          'apiServerURL':  'https://' + self.gke_attributes['endpoint'],
                          'username':      self.get_attribute('master_auth.0.username'),
                          'tlsCert' :      self.get_attribute('master_auth.0.client_certificate', True),
                          'tlsPrivateKey': self.get_attribute('master_auth.0.client_key', True),
                          'caCert':        ca_cert}
            if self.get_attribute('master_auth.0.password'):
                properties['password'] = self.get_attribute('master_auth.0.password')
            return [
                super(GKEClusterMapper, self)._create_ci("k8s.Master", cluster_id, properties),
                self.create_namespace(cluster_id, "default")
            ]
        else:
            return None

    def create_namespace(self, cluster_id, name):
        namespace_id = "%s/%s" %(cluster_id, name)
        return super(GKEClusterMapper, self)._create_ci("k8s.Namespace", namespace_id, {'namespaceName': name})

    def get_attribute(self, key, b64decode=False):
        if key in self.gke_attributes:
            if b64decode:
                return base64.b64decode(self.gke_attributes[key])
            else:
                return self.gke_attributes[key]
        return None