package com.xebialabs.deployit.plugin.xlrelease;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import com.xebialabs.deployit.deployment.planner.DefaultDelta;
import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.generic.deployed.ExecutedScript;

import static com.google.common.collect.Sets.newHashSet;
import static com.google.common.collect.Sets.newLinkedHashSet;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.CREATE;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.DESTROY;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.MODIFY;

public class RecreateDataContributor {

    static final Type INSTALLATION = Type.valueOf("xlrelease.XLReleaseInstallation");
    private static final Set<Type> DATA_TYPES = newHashSet(Type.valueOf("xlrelease.InstalledXLReleaseData"), Type.valueOf("xlrelease.User"));

    @Contributor
    public static void insertDataOnReinstall(Deltas deltas, DeploymentPlanningContext plan) {
        for (Container container : affectedContainers(deltas)) {
            Boolean cleanRepositoryOnUpgrade = container.getProperty("cleanRepositoryOnUpgrade");
            if (cleanRepositoryOnUpgrade) {
                Set<Deployed<?, ?>> deployedData = getDeployedData(container, deltas, plan);
                recreateData(plan, deployedData);
            }
        }
    }

    private static Set<Container> affectedContainers(Deltas deltas) {
        Set<Container> containers = newHashSet();
        for (Delta delta : deltas.getDeltas()) {
            if (isServerInstallation(delta)) {
                containers.add(delta.getDeployed().getContainer());
            }
        }
        return containers;
    }

    private static void recreateData(DeploymentPlanningContext plan,  Set<Deployed<?, ?>> deployedData) {
        for (Deployed<?, ?> deployed : deployedData) {
            addCreateDataStep(deployed, plan);
        }
    }

    private static Set<Deployed<?, ?>> getDeployedData(Container container, Deltas deltas, DeploymentPlanningContext plan) {
        Set<Deployed<?, ?>> deployedData = getDeployedDataInContainer(container, plan.getRepository());
        deployedData.removeAll(getRemovedData(deltas));
        return deployedData;
    }

    private static void addCreateDataStep(Deployed<?, ?> deployed, DeploymentPlanningContext plan) {
        ExecutedScript<?> script = (ExecutedScript<?>) deployed;
        Delta delta = new DefaultDelta(Operation.MODIFY, deployed, deployed);
        script.executeCreate(plan, delta);
    }

    private static Set<Deployed<?, ?>> getDeployedDataInContainer(Container container, Repository repository) {
        Set<Deployed<?, ?>> data = newLinkedHashSet();
        for (Type type : DATA_TYPES) {
            List<Deployed<?, ?>> deployeds = repository.search(type, container.getId());
            data.addAll(deployeds);
        }
        return data;
    }

    private static Collection<Deployed<?, ?>> getRemovedData(Deltas deltas) {
        Set<Deployed<?, ?>> removed = newHashSet();
        for (Delta delta : deltas.getDeltas()) {
            if (isDataRemoval(delta)) {
                removed.add(delta.getPrevious());
            }
        }
        return removed;
    }

    private static boolean isServerInstallation(Delta delta) {
        if (delta.getDeployed() == null) {
            return false;
        }
        return delta.getDeployed().getType().equals(INSTALLATION) && (delta.getOperation() == CREATE || delta.getOperation() == MODIFY);
    }

    private static boolean isDataRemoval(Delta delta) {
        if (delta.getPrevious() == null) {
            return false;
        }
        return DATA_TYPES.contains(delta.getPrevious().getType()) && (delta.getOperation() == DESTROY);
    }

}
