package com.xebialabs.deployit.provision
package steps

import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Preview, PreviewStep, StepExitCode}
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, Scope, StepMetadata, StepPostConstructContext}
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, Container, Environment}
import com.xebialabs.deployit.provision.resolver.TemplateResolver
import com.xebialabs.deployit.provision.resolver.TemplateResolver.ResolutionResult
import com.xebialabs.deployit.repository.ChangeSet

import scala.collection.convert.ImplicitConversions._
import scala.reflect.ClassTag
import scala.util.{Failure, Success}

@StepMetadata(name = "create-all-provisioned-cis")
class CreateAllProvisionedItems extends ProvisionedItemsStep with PreviewStep {

  @transient
  private lazy val templateResolver = TemplateResolver(deployedApplication)


  override def validScope: Scope = Scope.POST_PLAN

  @RulePostConstruct
  def validateAndSetParameters(ctx: StepPostConstructContext) {
    validate(ctx)
    if (description == null || description.isEmpty) {
      description = s"Register containers and dictionaries in ${correctDeployedApplication(ctx).getEnvironment.getName}"
    }
    order = if (order != null) order else 75
    setParameters(ctx)
  }

  override def getPreview: Preview = {
    val deployedItems = resolveTemplatesOnPackage() ++ resolvedTemplatesPerDeployed().values.flatten
    val previewMessages = deployedItems.map {
      case ResolutionResult(_, Success(id), _, _) =>
        s"Create or update $id"
      case ResolutionResult(templateId, Failure(ex), _, _) =>
        s"FAILED on $templateId with message: ${ex.getMessage}"
    }
    Preview.withContents(s"Register containers and dictionaries in ${deployedApplication.getEnvironment.getName}\n" + previewMessages.mkString("\n"))
  }

  override def execute(ctx: ExecutionContext): StepExitCode = {
    implicit val changeSet = new ChangeSet
    implicit val theCtx = ctx
    ctx.logOutput("The environment is being provisioned")
    resolveEnvironmentAndTemplates
    repositoryService.execute(changeSet)
    StepExitCode.SUCCESS
  }

  @SuppressWarnings(Array("all")) // FIXME: Try.get is considered unsafe
  private def resolveEnvironmentAndTemplates(implicit ctx: ExecutionContext, changeSet: ChangeSet) = {
    val cisOnPackage = resolveTemplatesOnPackage()
    val cisPerDeployed = resolvedTemplatesPerDeployed()
    val allResolvedResults = cisOnPackage ++ cisPerDeployed.values.flatten
    allResolvedResults.foreach(ci => ctx.logOutput(s"Resolving ${ci.templateId}"))
    throwErrorOnFailedResolution(allResolvedResults)
    setBoundConfigurationItemsOnDeployeds(cisPerDeployed)
    setBoundConfigurationItemsOnBlueprint(cisOnPackage)
    val allConfigurationItems = allResolvedResults.map(_.configurationItem.get)
    val members: JSet[Container] = new JHashSet[Container](filterConfigurationItems[Container](allConfigurationItems))

    val environment: Environment = getEnvironment
    environment.setMembers(environment.getMembers ++ members)

    val dictionaries = new JArrayList(filterConfigurationItems[UdmDictionaryType](allConfigurationItems))
    val allDicts: JList[UdmDictionaryType] = environment.getDictionaries ++ dictionaries
    environment.setDictionaries(allDicts)
    changeSet.createOrUpdate(environment)
  }

  @SuppressWarnings(Array("all")) // FIXME: Try.get is considered unsafe
  private def throwErrorOnFailedResolution(resolvedItems: Set[ResolutionResult])(implicit ctx: ExecutionContext) = {
    val failed = resolvedItems.filter(_.configurationItem.isFailure)
    failed.foreach {result =>
      ctx.logError(s"Could not resolve ${result.templateId}. Cause: ${result.configurationItem.failed.get.getMessage}")
    }
    if (failed.nonEmpty) throw new RuntimeException("Not all templates have been resolved")
  }

  private[steps] def resolveTemplatesOnPackage(): Set[ResolutionResult] = {
    templateResolver.resolveBoundTemplate(deploymentPackage)
  }

  @SuppressWarnings(Array("all")) // FIXME: Try.get is considered unsafe
  private def setBoundConfigurationItemsOnBlueprint(resolvedCIs: Set[ResolutionResult])(implicit changeSet: ChangeSet): Unit = {
    val configurationItems = resolvedCIs.map(_.configurationItem.get)
    changeSet.createOrUpdate(new JArrayList(configurationItems))
    deployedApplication.setBoundConfigurationItems(new JHashSet(configurationItems))
  }

  @SuppressWarnings(Array("all")) // FIXME: Try.get is considered unsafe
  private def setBoundConfigurationItemsOnDeployeds(resolvedDeployeds: Map[DeployedType, Set[ResolutionResult]])(implicit changeSet: ChangeSet) = {
    resolvedDeployeds.foreach { case (p, results) =>
      val configurationItems = results.map(_.configurationItem.get)
      changeSet.createOrUpdate(new JArrayList(configurationItems))
      p.setBoundConfigurationItems(new JHashSet(configurationItems))
    }
  }

  private[steps] def resolvedTemplatesPerDeployed(): Map[DeployedType, Set[ResolutionResult]] = {
    deployeds.map(p => p -> resolveTemplatesOnDeployed(p)).toMap
  }

  private[steps] def resolveTemplatesOnDeployed(deployed: DeployedType): Set[ResolutionResult] = {
    templateResolver.resolveBoundTemplate(deployed)
  }

  private[this] def filterConfigurationItems[C <: ConfigurationItem : ClassTag](boundCis: Iterable[ConfigurationItem]): Iterable[C] = {
    boundCis.withFilter(ci => ci.getType.getDescriptor.isAssignableTo(implicitly[ClassTag[C]].runtimeClass)).map(_.asInstanceOf[C])
  }
}
