package com.xebialabs.deployit.provision.steps

import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Preview, PreviewStep, StepExitCode}
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, Scope, StepMetadata, StepPostConstructContext}
import com.xebialabs.deployit.provision.RefMap
import com.xebialabs.deployit.provision.repository.SqlReferenceService
import com.xebialabs.deployit.repository.ChangeSet
import com.xebialabs.deployit.util.Tuple
import grizzled.slf4j.Logging

import scala.collection.convert.ImplicitConversions._

@StepMetadata(name = "validate-deprovisioning")
class ValidateDeprovisioningStep extends ProvisionedItemsStep with PreviewStep with Logging {

  override def validScope: Scope = Scope.PRE_PLAN

  @RulePostConstruct
  def validateAndSetParameters(ctx: StepPostConstructContext) {
    validate(ctx)
    if (description == null || description.isEmpty) {
      description = "Validate all configuration items to be deprovisioned"
    }
    order = if (order != null) order else 15
    setParameters(ctx)
  }

  override def getPreview: Preview = {
    Preview.withContents(s"Validating all configuration items to be deprovisioned:\n${allCiToDelete.mkString("\n")}")
  }

  override def execute(ctx: ExecutionContext): StepExitCode = {
    ctx.logOutput("Building changeset for validating configuration items to be deprovisioned")
    val changeSet = new ChangeSet
    val ciIds = allCiToDelete
    changeSet.delete(ciIds.map(Tuple.of(null: Integer, _)).toList)
    val references = computeNodeReferences(changeSet, deployedApplication.getEnvironment.getId, ciIds)
    references match {
      case Nil =>
        ctx.logOutput("Validation successful")
        StepExitCode.SUCCESS
      case _ =>
        ctx.logError(s"Configuration Item(s) ${references.mkString(" , ")} to be deleted are being referenced by either an environment or a deployed")
        StepExitCode.FAIL
    }
  }

  private def computeNodeReferences(changeSet: ChangeSet, envId: String, ciIds: Set[String]): List[String] = {
    val nodeReferences = referenceService.findCiReferences(changeSet)
    nodeReferences.map {
      case (k, v) => k -> v.filter(id => {
        val ciId = id.substring(1)
        !ciId.equals(envId) && !ciIds(ciId)
      })
    }.filter(_._2.nonEmpty).keys.toList
  }

  private def allCiToDelete: Set[String] = {
    resolvedTemplateOnDeployedApplication ++ resolvedTemplatesOnDeployeds
  }

  private def resolvedTemplateOnDeployedApplication: Set[String] = {
    deployedApplication.getBoundConfigurationItems.map(_.getId).toSet
  }

  private def resolvedTemplatesOnDeployeds: Set[String] = {
    deployeds.flatMap(_.getBoundConfigurationItems.toSet).map(_.getId)
  }

  protected def referenceService: ReferenceService = new SqlReferenceService

}

trait ReferenceService {
  def findCiReferences(changeSet: ChangeSet): RefMap
}
