/**
 * Copyright 2014-2018 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.core.rest.api

import ai.digital.configuration.central.deploy.converter.HoconDurationConverter
import akka.actor.ActorSystem
import akka.util.Timeout
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource
import com.xebialabs.deployit.engine.api.dto.{ConfigurationItemId, Paging, SatelliteInfo}
import com.xebialabs.deployit.engine.api.{SatelliteService, dto}
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.satellite.SatelliteInfoChecker.Get
import com.xebialabs.deployit.plugin.satellite._
import com.xebialabs.deployit.plugin.satellite.extension.FileSystemExtensionsLocator
import com.xebialabs.deployit.provision.JList
import com.xebialabs.deployit.repository.{RepositoryService, SatelliteRepository}
import com.xebialabs.deployit.security.permission.PlatformPermissions.READ
import com.xebialabs.deployit.tasksystem.TaskActorSystem
import com.xebialabs.satellite.protocol.Paths
import com.xebialabs.xlplatform.satellite.Satellite
import grizzled.slf4j.Logging
import org.jboss.resteasy.spi.HttpResponse
import org.springframework.beans.factory.annotation.{Autowired, Value}
import org.springframework.stereotype.Service
import java.util

import ai.digital.deploy.sql.http.enricher.PaginationService
import javax.ws.rs.core.Context

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{Await, ExecutionContextExecutor, Future}
import scala.jdk.CollectionConverters._

@Service
class SatelliteServiceImpl @Autowired()(satelliteRepository: SatelliteRepository,
                                        paginationService: PaginationService,
                                        repositoryService: RepositoryService)
  extends AbstractSecuredResource with SatelliteService with Logging {

  @Context val response: HttpResponse = null

  @Value("${deploy.satellite.timeout.info:10 seconds}")
  var defaultPingTimeoutProp: String = _

  private def defaultPingTimeout: FiniteDuration = HoconDurationConverter.convert(defaultPingTimeoutProp)

  override def getInfo(ciId: String): SatelliteInfo = {
    import akka.pattern._
    implicit val actorSystem: ActorSystem = TaskActorSystem.actorSystem
    implicit val dispatcher: ExecutionContextExecutor = actorSystem.dispatcher
    implicit val timeout: Timeout = Timeout(defaultPingTimeout)

    val ci: ConfigurationItem = repositoryService.read(ciId)
    if (!ci.getType.instanceOf(Type.valueOf("xl.Satellite"))) {
      return null
    }

    val satellite: Satellite = ci.asInstanceOf[Satellite]
    val actorLocator = ActorLocator(satellite)
    val checker = TaskActorSystem.actorSystem.actorOf(SatelliteInfoChecker.props(satellite.getAddress))

    val future = (checker ? Get(actorLocator.locate(Paths.info))).recover {
      case error =>
        logger.debug(s"Operation failed on satellite ${satellite.getAddress} (${error.getMessage})")
        new SatelliteInfo()
    }

    val satelliteInfo = Await.result(future, 1.5 * defaultPingTimeout).asInstanceOf[SatelliteInfo]
    if (satelliteInfo.isAvailable()) {
      arePluginsSynced(ci.getId) match {
        case Some(pluginSync) =>
          satelliteInfo.setPluginsSynced(pluginSync)
        case None =>
      }
    }

    satelliteInfo.setSatelliteId(ciId)

    satelliteInfo
  }

  override def getInfos(ids: JList[String]): JList[SatelliteInfo] = {
    implicit val dispatcher: ExecutionContextExecutor = TaskActorSystem.actorSystem.dispatcher
    val getInfoFutures = ids.asScala.map(ciId => Future {
      getInfo(ciId)
    })
    Await.result(Future.sequence(getInfoFutures), 3 * defaultPingTimeout).toList.asJava
  }

  private def arePluginsSynced(ciId: String): Option[Boolean] = {
    implicit val actorSystem: ActorSystem = TaskActorSystem.actorSystem
    implicit val timeout: Timeout = Timeout(defaultPingTimeout)

    val ci: ConfigurationItem = repositoryService.read(ciId)
    if (!ci.getType.instanceOf(Type.valueOf("xl.Satellite"))) {
      return None
    }

    val satellite: Satellite = ci.asInstanceOf[Satellite]
    val actorLocator = ActorLocator(satellite)

    try {
      val deltas = ExtensionsDelta.calculateExtensionsDelta(FileSystemExtensionsLocator, actorLocator, actorSystem)
      Some(deltas.empty)
    } catch {
      case error: Exception =>
        logger.error(s"Plugin sync check operation failed on satellite ${satellite.getAddress} (${error.getMessage})")
        Option.empty[Boolean]
    }
  }

  override def listHosts(ciId: String, name: String, paging: Paging, order: dto.Ordering): util.List[ConfigurationItemId] = {
    checkPermission(READ, ciId)
    paginationService.addPagingHeaderIfNeeded(paginationService.toSetHeader(response), () => countHosts(ciId, name), paging)
    satelliteRepository.listHosts(ciId, name, paging, order)
  }

  override def countHosts(ciId: String, name: String): Integer = {
    checkPermission(READ, ciId)
    satelliteRepository.countHosts(ciId, name)
  }
}
