package com.xebialabs.deployit.core.rest.api;

import ai.digital.deploy.sql.model.Report;
import ai.digital.deploy.sql.model.ReportLine;
import com.xebialabs.deployit.core.api.InternalReportProxy;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.core.converters.ControlTaskReportLineConverter;
import com.xebialabs.deployit.core.converters.DeploymentTaskReportLineConverter;
import com.xebialabs.deployit.core.rest.api.reports.AggregatedDeploymentsKeyIndicator;
import com.xebialabs.deployit.core.rest.api.reports.DeploymentsForEnvironmentReport;
import com.xebialabs.deployit.core.rest.api.reports.DeploymentsKeyIndicator;
import com.xebialabs.deployit.core.rest.api.reports.DeploymentsStateBreakdownReport;
import com.xebialabs.deployit.core.rest.api.reports.widgets.WidgetRegistry;
import com.xebialabs.deployit.core.rest.resteasy.Workdir;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.Ordering;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.deployit.engine.api.execution.BlockState;
import com.xebialabs.deployit.engine.api.execution.StepBlockState;
import com.xebialabs.deployit.engine.api.execution.TaskWithBlock;
import com.xebialabs.deployit.engine.api.execution.TaskWithSteps;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.FilterType;
import com.xebialabs.deployit.task.archive.TaskArchive;
import com.xebialabs.deployit.task.archive.TaskReader;
import org.jboss.resteasy.spi.HttpResponse;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.ws.rs.core.Context;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static ai.digital.deploy.tasker.common.TaskType.CONTROL;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.deployit.core.rest.api.reports.ReportUtils.DATE_FORMAT;
import static com.xebialabs.deployit.core.rest.api.reports.ReportUtils.createDownloadToken;
import static com.xebialabs.deployit.core.rest.resteasy.Workdir.Clean.ALWAYS;
import static com.xebialabs.deployit.repository.WorkDirFactory.DOWNLOAD_WORKDIR_PREFIX;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.REPORT_VIEW;
import static java.lang.String.format;
import static java.util.EnumSet.of;

@Service
public class ReportResource extends AbstractTaskRestrictedResource implements InternalReportProxy {
    @Context
    private HttpResponse response;

    @Autowired
    private DownloadResource downloadResource;

    @Transactional(value = "reportingTransactionManager", readOnly = true)
    @Override
    public Report deploymentsForEnvironment(final String environment, final Date date) {
        checkPermission(REPORT_VIEW);

        checkNotNull(environment, "environment");
        checkNotNull(date, "date");
        checkNotNull(date.getCalendar(), "date");

        DeploymentsForEnvironmentReport report = new DeploymentsForEnvironmentReport(taskArchive);
        return report.report(environment, date.asDateTime());
    }

    @Workdir(prefix = DOWNLOAD_WORKDIR_PREFIX, clean = ALWAYS)
    @Transactional(value = "reportingTransactionManager", readOnly = true)
    @Override
    public String downloadDeploymentsForEnvironment(final String environment, final Date date) {
        checkPermission(REPORT_VIEW);
        Report report = deploymentsForEnvironment(environment, date);
        String fileName = format("deployments-%s-%s.csv", environment.replaceAll("/", "_"), DATE_FORMAT.print(date.asDateTime()));

        return createDownloadToken(downloadResource, report, fileName);
    }

    @Override
    @Deprecated
    public TaskWithSteps getTask(final String taskId) {
        return new TaskWithStepsView(retrieveTask(taskId, true));
    }

    @Override
    public TaskWithBlock getTaskWithBlock(final String taskId) {
        return retrieveTask(taskId, false);
    }

    @Override
    public TaskWithBlock getTaskWithBlockAndSteps(String taskId) {
        return retrieveTask(taskId, true);
    }

    private TaskWithBlock retrieveTask(final String taskId, boolean loadSteps) {
        checkPermission(REPORT_VIEW);
        TaskReader taskReader = taskArchive.getTask(taskId);
        TaskWithBlock t = loadSteps ? taskReader.fully() : taskReader.withoutSteps();
        checkAccessTo(t.getMetadata());
        return t;
    }

    @Override
    public StepBlockState getStepBlock(String taskId, String blockId) {
        checkPermission(REPORT_VIEW);
        TaskReader taskReader = taskArchive.getTask(taskId);
        checkAccessTo(taskReader.metadata());
        BlockState block = taskReader.block(blockId);
        if (block instanceof StepBlockState) {
            return (StepBlockState) block;
        } else {
            throw new DeployitException("Block [%s] from task [%s] is a composite block so it has no steps", blockId, taskId);
        }
    }

    @Override
    public Stream<ReportLine> getTaskReport(Date begin, Date end, Paging paging, List<Ordering> order, String filterType,
                                            List<String> users, List<String> states, String taskId, boolean onlySuccessful,
                                            String workerName, List<ConfigurationItemId> configurationItemIds) {
        Paging limited = paginationService.getLimitedPaging(paging);
        ArchivedTaskSearchParameters searchParameters = reportGenerator.buildFilteredTaskSearchParameters(begin, end,
                limited, order, filterType, users, states, null, null, taskId, configurationItemIds,
                ReportGenerator.DEPLOYMENT_ALLOWED_SORT_FIELDS, onlySuccessful, workerName);
        paginationService.addPagingHeaderIfNeeded(paginationService.toSetHeader(response),
                () -> taskArchive.countTotalResults(searchParameters), paging);
        return reportGenerator.streamTaskReport(searchParameters).map(DeploymentTaskReportLineConverter::toReportLine);
    }

    @Workdir(prefix = DOWNLOAD_WORKDIR_PREFIX, clean = ALWAYS)
    @Transactional(value = "reportingTransactionManager", readOnly = true)
    @Override
    public String downloadTaskReport(final Date begin, final Date end, final String filterType,
                                     List<String> users, List<String> states, String taskId, boolean onlySuccessful,
                                     String workerName, final List<ConfigurationItemId> configurationItemIds) {
        ArchivedTaskSearchParameters parameters = reportGenerator.buildFilteredTaskSearchParameters(begin, end, new Paging(),
                Collections.emptyList(), filterType, users, states, null, null, taskId, configurationItemIds,
                ReportGenerator.DEPLOYMENT_ALLOWED_SORT_FIELDS, onlySuccessful, workerName);
        Stream<ReportLine> report = reportGenerator.streamTaskReport(parameters).map(DeploymentTaskReportLineConverter::toReportLine);
        String fileName = format("tasks-%s-%s.csv", DATE_FORMAT.print(begin.asDateTime()), DATE_FORMAT.print(end.asDateTime()));
        return createDownloadToken(downloadResource, report, fileName);
    }

    @Override
    public Report deploymentsStateBreakdown(String filterType, Date begin, Date end, List<ConfigurationItemId> configurationItemIds) {
        checkNotNull(filterType, "filterType");
        checkNotNull(begin, "begin");
        checkNotNull(begin.getCalendar(), "begin");
        checkNotNull(end, "end");
        checkNotNull(end.getCalendar(), "end");
        checkNotNull(configurationItemIds, "configurationItemIds");

        DeploymentsStateBreakdownReport report = new DeploymentsStateBreakdownReport(taskArchive);
        FilterType reportFilterType = FilterType.valueOf(filterType.toUpperCase());
        return report.report(reportFilterType, configurationItemIds);
    }

    @Workdir(prefix = DOWNLOAD_WORKDIR_PREFIX, clean = ALWAYS)
    @Override
    public String downloadDeploymentsStateBreakdown(final String filterType, final Date begin, final Date end, final List<ConfigurationItemId> configurationItemIds) {
        checkPermission(REPORT_VIEW);
        Report report = deploymentsStateBreakdown(filterType, begin, end, configurationItemIds);
        String fileName = format("deployments-by-state-%s-%s.csv", DATE_FORMAT.print(begin.asDateTime()), DATE_FORMAT.print(end.asDateTime()));

        return createDownloadToken(downloadResource, report, fileName);
    }

    @Transactional(value = "reportingTransactionManager", readOnly = true)
    @Override
    public Report deploymentsKeyIndicator(Date begin, Date end, String filterType, List<ConfigurationItemId> configurationItemIds) {
        checkPermission(REPORT_VIEW);

        checkNotNull(begin, "begin");
        checkNotNull(begin.getCalendar(), "begin");
        checkNotNull(end, "end");
        checkNotNull(end.getCalendar(), "end");

        DeploymentsKeyIndicator report = new DeploymentsKeyIndicator(taskArchive);
        report.getSearchParameters().createdBetween(begin.asDateTime(), end.asDateTime());
        if (filterType != null) {
            FilterType filter = FilterType.valueOf(filterType.toUpperCase());
            if (filter != FilterType.NONE) {
                report.getSearchParameters().forFilterType(filter, configurationItemIds);
            }
        }

        return report.report();
    }

    @Override
    public Stream<ReportLine> getControlTasksReport(Date begin, Date end, Paging paging, List<Ordering> order) {
        Paging limited = paginationService.getLimitedPaging(paging);
        ArchivedTaskSearchParameters searchParameters = reportGenerator.buildTaskSearchParameters(limited, order, of(CONTROL),
                ReportGenerator.CONTROL_ALLOWED_SORT_FIELDS);
        searchParameters.inDateTimeRange(reportGenerator.safeDateTime(begin, null), reportGenerator.safeDateTime(end, null));
        paginationService.addPagingHeaderIfNeeded(paginationService.toSetHeader(response),
                () -> taskArchive.countTotalResults(searchParameters), paging);
        return reportGenerator.streamControlTasksReport(searchParameters).map(ControlTaskReportLineConverter::toReportLine);
    }

    @Workdir(prefix = DOWNLOAD_WORKDIR_PREFIX, clean = ALWAYS)
    @Transactional(value = "reportingTransactionManager", readOnly = true)
    @Override
    public String downloadControlTasksReport(final Date begin, final Date end) {
        ArchivedTaskSearchParameters searchParameters = reportGenerator.buildTaskSearchParameters(new Paging(), Collections.emptyList(), of(CONTROL),
                ReportGenerator.CONTROL_ALLOWED_SORT_FIELDS);
        searchParameters.inDateTimeRange(reportGenerator.safeDateTime(begin, null), reportGenerator.safeDateTime(end, null));
        Stream<ReportLine> report = reportGenerator.streamControlTasksReport(searchParameters).map(ControlTaskReportLineConverter::toReportLine);
        return createDownloadToken(downloadResource, report, reportGenerator.buildControlTaskReportFileName(begin, end));
    }

    @Transactional(value = "reportingTransactionManager", readOnly = true)
    @Override
    public Report deploymentsKeyIndicatorAggregatedByFilterType(String filterType, Date begin, Date end, List<ConfigurationItemId> configurationItemIds) {
        checkPermission(REPORT_VIEW);

        checkNotNull(filterType, "filterType");
        checkNotNull(begin, "begin");
        checkNotNull(begin.getCalendar(), "begin");
        checkNotNull(end, "end");
        checkNotNull(end.getCalendar(), "end");
        checkNotNull(configurationItemIds, "configurationItemIds");

        FilterType reportFilterType = FilterType.valueOf(filterType.toUpperCase());

        AggregatedDeploymentsKeyIndicator report = new AggregatedDeploymentsKeyIndicator(taskArchive);
        report.getSearchParameters().createdBetween(new DateTime(begin.getCalendar()), new DateTime(end.getCalendar()));

        return report.report(reportFilterType, configurationItemIds);
    }

    @Workdir(prefix = DOWNLOAD_WORKDIR_PREFIX, clean = ALWAYS)
    @Transactional(value = "reportingTransactionManager", readOnly = true)
    @Override
    public String downloadDeploymentsKeyIndicator(final String filterType, final Date begin, final Date end, final List<ConfigurationItemId> cis) {
        checkPermission(REPORT_VIEW);
        Report report = deploymentsKeyIndicator(begin, end, filterType, cis);
        String fileName = format("deployments-key-indicator-%s-%s.csv", DATE_FORMAT.print(begin.asDateTime()), DATE_FORMAT.print(end.asDateTime()));

        return createDownloadToken(downloadResource, report, fileName);
    }

    @Workdir(prefix = DOWNLOAD_WORKDIR_PREFIX, clean = ALWAYS)
    @Transactional(value = "reportingTransactionManager", readOnly = true)
    @Override
    public String downloadDeploymentsKeyIndicatorAggregatedByFilterType(final String filterType, final Date begin, final Date end, final List<ConfigurationItemId> configurationItemIds) {
        checkPermission(REPORT_VIEW);
        Report report = deploymentsKeyIndicatorAggregatedByFilterType(filterType, begin, end, configurationItemIds);
        String fileName = format("aggregated-deployments-key-indicator-%s-%s-%s.csv", filterType, DATE_FORMAT.print(begin.asDateTime()),
                DATE_FORMAT.print(end.asDateTime()));

        return createDownloadToken(downloadResource, report, fileName);
    }

    @Transactional(value = "reportingTransactionManager", readOnly = true)
    @Override
    public Report getDashboardWidgetReport(String widget, Date begin, Date end) {
        checkPermission(REPORT_VIEW);

        checkNotNull(widget, "widget");
        checkNotNull(begin, "begin");
        checkNotNull(begin.getCalendar(), "begin");
        checkNotNull(end, "end");
        checkNotNull(end.getCalendar(), "end");
        checkArgument(begin.getCalendar().getTime().compareTo(end.getCalendar().getTime()) <= 0, "begin must be before end");

        return WidgetRegistry.valueOf(WidgetRegistry.class, widget.toUpperCase()).getWidget(taskArchive).getReport(begin.asDateTime(), end.asDateTime());
    }

    @Override
    public List<ConfigurationItemId> listEnvironments() {
        checkPermission(REPORT_VIEW);

        List<String> environments = taskArchive.getAllArchivedEnvironments();
        Collections.sort(environments);

        return environments.stream().map(input -> new ConfigurationItemId(input, Type.valueOf("udm.Environment"))).collect(Collectors.toList());
    }

    public void setTaskArchive(final TaskArchive taskArchive) {
        this.taskArchive = taskArchive;
    }

    public void setRepositoryService(final RepositoryService repositoryService) {
        this.repositoryService = repositoryService;
    }

    public void setDownloadResource(DownloadResource downloadResource) {
        this.downloadResource = downloadResource;
    }

    public void setResponse(HttpResponse response) {
        this.response = response;
    }
}
