package com.xebialabs.deployit.core.rest.websockets

import com.xebialabs.deployit.core.events.{CiRef, JmsCiChangeSetEvent}
import com.xebialabs.deployit.core.websockets.WebSocketEndpoints
import com.xebialabs.deployit.engine.spi.event.{MaintenanceStartEvent, MaintenanceStopEvent}
import com.xebialabs.deployit.event.EventBusHolder
import nl.javadude.t2bus.Subscribe
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.messaging.simp.SimpMessagingTemplate
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.stereotype.Controller

import java.util.{Collection => JavaCollection}
import scala.jdk.CollectionConverters._

@Autowired
@Controller
class WebSocketMessagesSender(@Autowired messageTemplate: SimpMessagingTemplate) {
  EventBusHolder.register(this)

  @Subscribe
  def sendWsCiChangeSetMessages(event: JmsCiChangeSetEvent): Unit = {
    notifyCisIfNotEmpty(WebSocketEndpoints.TOPIC_CI_CREATE, event.createCis.asJava)
    notifyCisIfNotEmpty(WebSocketEndpoints.TOPIC_CI_UPDATE, event.updateCis.asJava)
    notifyWithItemsIfNotEmpty(WebSocketEndpoints.TOPIC_CI_DELETE, event.deleteCis.asJava)
    notifyWithItemsIfNotEmpty(WebSocketEndpoints.TOPIC_CI_RENAME, event.renameCis.asJava)
    notifyWithItemsIfNotEmpty(WebSocketEndpoints.TOPIC_CI_MOVE, event.moveCis.asJava)
    notifyWithItemsIfNotEmpty(WebSocketEndpoints.TOPIC_CI_COPY, event.copyCis.asJava)
  }

  @Subscribe
  def sendMaintenanceStarted(event: MaintenanceStartEvent): Unit = {
    notifyAboutMaintenanceModeChange(true)
  }

  @Subscribe
  def sendMaintenanceStopped(event: MaintenanceStopEvent): Unit = {
    notifyAboutMaintenanceModeChange(false)
  }

  private def notifyAboutMaintenanceModeChange(enabled: Boolean): Unit = {
    messageTemplate.convertAndSend(WebSocketEndpoints.TOPIC_MAINTENANCE.getName, enabled)
  }

  private def getUserName = {
    Option(SecurityContextHolder.getContext.getAuthentication).map(_.getName).getOrElse("admin")
  }

  private def notifyCisIfNotEmpty(endpoint: WebSocketEndpoints, ciIds: JavaCollection[CiRef]): Unit = {
    notifyWithItemsIfNotEmpty(endpoint, ciIds)
  }

  private def notifyWithItemsIfNotEmpty[T](endpoint: WebSocketEndpoints, items: JavaCollection[T]): Unit = {
    if (!items.isEmpty) {
      messageTemplate.convertAndSendToUser(getUserName, endpoint.getName, items)
    }
  }

}
