package com.xebialabs.deployit.core.rest.json;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.io.CharStreams;
import com.xebialabs.deployit.core.util.ReaderWriterUtils;
import com.xebialabs.deployit.core.xml.PasswordEncryptingCiConverter;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xltype.serialization.CiWriter;
import com.xebialabs.xltype.serialization.json.CiJsonWriter;
import com.xebialabs.xltype.serialization.json.CiListJsonReader;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.ext.MessageBodyReader;
import jakarta.ws.rs.ext.MessageBodyWriter;
import jakarta.ws.rs.ext.Provider;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Collection;

@Component
@Provider
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
public class CisJsonMessageBodyReaderWriter implements MessageBodyWriter<Collection<ConfigurationItem>>, MessageBodyReader<Collection<ConfigurationItem>> {
    private final ReaderPostProcessor postProcessor;

    private final JsonWriterHelper helper = new JsonWriterHelper();

    @Autowired
    public CisJsonMessageBodyReaderWriter(ReaderPostProcessor postProcessor) {
        this.postProcessor = postProcessor;
    }

    @Override
    public boolean isReadable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return isWriteable(type, genericType, annotations, mediaType);
    }

    @Override
    public boolean isWriteable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        if (!mediaType.toString().contains(MediaType.APPLICATION_JSON)) {
            return false;
        }

        return ConfigurationItem.class.isAssignableFrom(helper.getType(type, genericType));
    }

    @Override
    public long getSize(Collection<ConfigurationItem> t, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return -1L;
    }

    @Override
    public void writeTo(Collection<ConfigurationItem> cis, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, Object> httpHeaders,
        OutputStream entityStream) throws IOException, WebApplicationException {
        entityStream.write(toJson(cis).getBytes());
    }

    @Override
    public Collection<ConfigurationItem> readFrom(Class<Collection<ConfigurationItem>> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, String> httpHeaders, InputStream entityStream) throws IOException, WebApplicationException {
        return toObject(toString(entityStream), ReaderWriterUtils.getCiTreeDepth(annotations));
    }

    @VisibleForTesting
    static String toJson(Collection<ConfigurationItem> cis) {
        PasswordEncryptingCiConverter converter = getConverter();
        CiWriter writer = new CiJsonWriter();
        writer.startList();
        cis.forEach(configurationItem -> converter.writeCi(configurationItem, writer));
        writer.endList();

        return writer.toString();
    }

    @VisibleForTesting
    Collection<ConfigurationItem> toObject(String json, Integer depth) {
        PasswordEncryptingCiConverter converter = getConverter();
        return postProcessor.apply(context -> {
            context.register(converter);
            return converter.readCis(CiListJsonReader.create(json));
        }, depth);
    }

    private static PasswordEncryptingCiConverter getConverter() {
        return new PasswordEncryptingCiConverter();
    }

    public static String toString(InputStream in) throws IOException {
        return CharStreams.toString(new InputStreamReader(in, Charsets.UTF_8));
    }
}
