package com.xebialabs.deployit.core.rest.api;


import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static com.google.common.io.ByteStreams.newInputStreamSupplier;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.core.api.dto.Artifact;
import com.xebialabs.deployit.core.api.dto.ArtifactAndData;
import com.xebialabs.deployit.core.api.dto.ConfigurationItem;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyKind;
import com.xebialabs.deployit.core.api.dto.FullTaskInfo;
import com.xebialabs.deployit.core.api.dto.Message;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.core.api.dto.RepositoryObjects;
import com.xebialabs.deployit.core.api.dto.StepInfo;
import com.xebialabs.deployit.core.api.dto.Steps;
import com.xebialabs.deployit.core.api.dto.TaskInfo;
import com.xebialabs.deployit.core.api.dto.TaskInfos;
import com.xebialabs.deployit.plugin.PojoConverter;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.repository.ArtifactEntity;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.repository.RepositoryObjectEntity;
import com.xebialabs.deployit.service.validation.ValidationMessage;
import com.xebialabs.deployit.task.DeploymentTask;
import com.xebialabs.deployit.task.DeploymentTaskInfo;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.TaskStepInfo;
import com.xebialabs.deployit.util.PasswordObfuscator;

/**
 */
@Component
public class DtoConverter {


    private PojoConverter pojoConverter;

    @Autowired
    public DtoConverter(PojoConverter pojoConverter) {
        this.pojoConverter = pojoConverter;
    }

    public RepositoryObject entityToDto(final RepositoryObjectEntity entity) {
        checkNotNull(entity, "entity");
        checkNotNull(entity.getId(), "entity.id");

        RepositoryObject dto;
        if (entity instanceof ArtifactEntity) {
            dto = new Artifact();
        } else if (entity instanceof ConfigurationItemEntity) {
            dto = new ConfigurationItem();
        } else {
            throw new IllegalArgumentException("entity is not of class " + ConfigurationItemEntity.class.getName() + " or class "
                    + ArtifactEntity.class.getName());
        }

        dto.setId(entity.getId());
        dto.setType(entity.getType().toString());
        dto.setLastModified(entity.getLastModified());
        dto.setOverrideLastModified(null); // Default value for DTO
        dto.setCreatingTaskId(entity.getCreatingTaskId());
        if(entity instanceof ArtifactEntity) {
            ((Artifact) dto).setFilename(((ArtifactEntity) entity).getFilename());
        }
        copyValuesToDto(entity, dto);
        return dto;
    }

    private void copyValuesToDto(final RepositoryObjectEntity entity, final RepositoryObject dto) {
        final HashMap<String, Object> map = Maps.newHashMap();
        for (Map.Entry<String, Object> entry : entity.getValues().entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();
            map.put(key, value);
        }

        dto.setValues(map);
    }

    public RepositoryObjectEntity entityFromDto(final RepositoryObject dto) {
        checkNotNull(dto, "dto");
        checkNotNull(dto.getId(), "dto.id");

        RepositoryObjectEntity entity = createEntity(dto);

        entity.setId(dto.getId());
        entity.setLastModified(null); // Cannot be overridden from DTO
        entity.setOverrideLastModified(dto.getOverrideLastModified());
        entity.setCreatingTaskId(null); // Cannot be overridden from DTO
        if(entity instanceof ArtifactEntity) {
            ((ArtifactEntity) entity).setFilename(((Artifact) dto).getFilename());
        }
        copyValues(dto, entity);
        return entity;
    }

    private void copyValues(RepositoryObject from, RepositoryObjectEntity to) {
        final Descriptor descriptor = DescriptorRegistry.getDescriptor(to.getType());
        for (String fromKey : from.getValues().keySet()) {
            PropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(fromKey);
            Object value = from.getValues().get(fromKey);
            if (propertyDescriptor != null && propertyDescriptor.isPassword()) {
                to.addValue(fromKey, PasswordObfuscator.ensureEncrypted(value.toString()));
            } else {
                to.addValue(fromKey, value);
            }
        }
    }

    private RepositoryObjectEntity createEntity(RepositoryObject dto) {
        RepositoryObjectEntity entity;
        if (dto instanceof Artifact) {
            entity = createObjectWithDefaults(dto.getType());
        } else if (dto instanceof ConfigurationItem) {
            entity = createObjectWithDefaults(dto.getType());
        } else {
	        throw new IllegalArgumentException("Do not know the type " + dto.getClass() + " to create entity for.");
        }
        return entity;
    }

	private ConfigurationItemEntity createObjectWithDefaults(String type) {
        checkNotNull(type, "dto.type");
        Descriptor descriptor = DescriptorRegistry.getDescriptor(type);
        checkArgument(descriptor != null, type + " is an unknown type");
        return pojoConverter.toEntity(descriptor.newInstance());
    }

    public ArtifactEntity artifactEntityFromDto(final ArtifactAndData dto) {
        checkNotNull(dto, "dto");
        checkNotNull(dto.getArtifact(), "dto.artifact");
        checkNotNull(dto.getArtifact().getId(), "dto.artifact.id");
        checkNotNull(dto.getData(), "dto.artifact.data");

        ArtifactEntity entity = (ArtifactEntity) entityFromDto(dto.getArtifact());
        entity.setData(newInputStreamSupplier(dto.getData()));
        return entity;
    }

    public RepositoryObjects entitiesToDto(final Collection<? extends RepositoryObjectEntity> entities) {
        checkNotNull(entities, "entities");

        final RepositoryObjects objects = new RepositoryObjects();
        for (RepositoryObjectEntity each : entities) {
            objects.add(entityToDto(each));
        }
        return objects;
    }

    public List<ConfigurationItem> deployedsToDto(final ListMultimap<Boolean, ConfigurationItemEntity> deployeds) {
        checkNotNull(deployeds, "deployeds");

		List<ConfigurationItem> cis = newArrayList();
        for (ConfigurationItemEntity validDeployedEntity : deployeds.get(true)) {
            ConfigurationItem validDeployed = (ConfigurationItem) entityToDto(validDeployedEntity);
            cis.add(validDeployed);
        }
        for (ConfigurationItemEntity invalidDeployedEntity : deployeds.get(false)) {
            ConfigurationItem invalidDeployed = (ConfigurationItem) entityToDto(invalidDeployedEntity);
            ArrayList<Message> validations = new ArrayList<Message>();
            validations.add(new Message("source", "The deployable for this deployed is missing from the package."));
            invalidDeployed.setValidations(validations);
            cis.add(invalidDeployed);
        }
        return cis;
    }

    @SuppressWarnings("unchecked")
    public <T extends RepositoryObjectEntity> Collection<T> entitiesFromDto(final List<? extends RepositoryObject> objects) {
        checkNotNull(objects, "objects");

        List<T> entities = newArrayList();
        for (RepositoryObject each : objects) {
            entities.add((T) entityFromDto(each));
        }
        return entities;
    }

    public ConfigurationItemDescriptorDto descriptorToDto(final Descriptor descriptor) {
        checkNotNull(descriptor, "descriptor");

        final ConfigurationItemDescriptorDto dto = new ConfigurationItemDescriptorDto();
        dto.setType(descriptor.getType().toString());
        dto.setSimpleName(descriptor.getType().toString()); // FIXME: should we still send the simple class name?
        Function<Type, String> typeToString = new Function<Type, String>() {
            public String apply(Type from) {
                return from.toString();
            }
        };
        dto.setInterfaces(newHashSet(Collections2.transform(descriptor.getInterfaces(), typeToString)));
        dto.setSuperClasses(newArrayList(Collections2.transform(descriptor.getSuperClasses(), typeToString)));
        Type deployableType = descriptor.getDeployableType();
        dto.setMappingSourceType(deployableType != null ? deployableType.toString() : "");
        Type containerType = descriptor.getContainerType();
        dto.setMappingTargetType(containerType != null ? containerType.toString() : "");
        dto.setDescription(descriptor.getDescription());
        dto.setPlaceholderFormatName(""); // FIXME: how do we handle placeholders in plugin-api-v3?
        dto.setPlaceholdersName(""); // FIXME: how do we handle placeholders in plugin-api-v3?
        dto.setRootName(descriptor.getRoot().getRootNodeName());
        for (PropertyDescriptor propertyDescriptor : descriptor.getPropertyDescriptors()) {
            if (!propertyDescriptor.isHidden()) {
                dto.addPropertyDescriptor(propertyDescriptorToDto(propertyDescriptor));
            }
        }

        return dto;
    }

    public ConfigurationItemPropertyDescriptorDto propertyDescriptorToDto(final PropertyDescriptor propertyDescriptor) {
        checkNotNull(propertyDescriptor, "propertydescriptor");

        final ConfigurationItemPropertyDescriptorDto dto = new ConfigurationItemPropertyDescriptorDto();
        dto.setName(propertyDescriptor.getName());
        dto.setLabel(propertyDescriptor.getLabel());
        dto.setType(ConfigurationItemPropertyKind.valueOf(propertyDescriptor.getKind().toString()));
        if (propertyDescriptor.getEnumValues() != null) {
            dto.setEnumValues(propertyDescriptor.getEnumValues().toArray(new String[propertyDescriptor.getEnumValues().size()]));
        }
        Type referencedType = propertyDescriptor.getReferencedType();
        dto.setCollectionMemberClassname(referencedType != null ? referencedType.toString() : "");
        dto.setPropertyClassname(""); // FIXME: for what was this used?
        dto.setRequired(propertyDescriptor.isRequired());
        dto.setEditable(true); // no read-only properties anymore
        dto.setPassword(propertyDescriptor.isPassword());
        dto.setIdentifying(false); // no identifying parameters anymore
        dto.setDiscoveryParam(false); // no discovery parameters anymore
        dto.setDiscoveryRequired(false); // no discovery parameters anymore
        dto.setCategory(propertyDescriptor.getCategory());
        dto.setSize(propertyDescriptor.getSize().toString());
        Object defaultValue = propertyDescriptor.getDefaultValue();
        dto.setDefaultValue(defaultValue == null ? null : defaultValue.toString());
        dto.setDescription(propertyDescriptor.getDescription());
        dto.setAsContainment(propertyDescriptor.isAsContainment());
        if (propertyDescriptor.getListDescriptors() != null) {
            for (PropertyDescriptor each : propertyDescriptor.getListDescriptors()) {
                dto.addPropertyDescriptor(propertyDescriptorToDto(each));
            }
        }
        return dto;
    }

    public TaskInfos tasksToDto(final List<Task> tasks) {
        TaskInfos taskInfos = new TaskInfos();
        for (Task eachTask : tasks) {
            taskInfos.add(taskToDto(eachTask.getId(), eachTask));
        }
        return taskInfos;
    }

   public TaskInfo taskToDto(String id, Task task) {
        checkNotNull(id, "id");
        checkNotNull(task, "task");

        TaskInfo ti = new TaskInfo();
        ti.setId(id);
        ti.setLabel(task.getLabel());
        ti.setState(task.getState().name());
        ti.setStartDate(task.getStartDate());
        ti.setCompletionDate(task.getCompletionDate());
        ti.setNrOfSteps(task.getNrOfSteps());
        ti.setCurrentStepNr(task.getCurrentStepNr());
        ti.setUser(task.getOwner());

        if (task instanceof DeploymentTask) {
            DeploymentTask dt = (DeploymentTask) task;
            ti.setApplication(dt.getApplicationName());
            ti.setVersion(dt.getApplicationVersion());
            ti.setEnvironment(dt.getEnvironment());
            ti.setFailureCount(dt.getFailureCount());
        }

        return ti;
    }

    public StepInfo taskStepInfoToDto(final int stepNr, final TaskStepInfo step) {
        checkNotNull(step, "step");

        StepInfo si = new StepInfo();
        si.setNr(stepNr);
        si.setDescription(step.getDescription());
        si.setState(step.getState().name());
        si.setStartDate(step.getStartDate());
        si.setCompletionDate(step.getCompletionDate());
        si.setLog(step.getLog());

        return si;
    }

    public Steps taskToDto(final Task task) {
        return taskStepInfosToDto(task.getSteps());
    }

    public Steps taskStepInfosToDto(final List<? extends TaskStepInfo> stepList) {
        Steps steps = new Steps();
        int stepNr = 1;
        for (TaskStepInfo taskStep : stepList) {
            steps.add(taskStepInfoToDto(stepNr++, taskStep));
        }
        return steps;
    }

    public TaskInfo archivedTaskToDto(DeploymentTaskInfo deploymentTask) {
        checkNotNull(deploymentTask, "deploymentTask");
        
        return populateTaskInfo(deploymentTask, new TaskInfo());
    }

	private <T extends TaskInfo> T populateTaskInfo(final DeploymentTaskInfo deploymentTask, final T taskInfo) {
		checkNotNull(deploymentTask, "deploymentTask");
		checkNotNull(taskInfo,"taskInfo");
		
		taskInfo.setId(deploymentTask.getId());
		taskInfo.setLabel(String.format("Deployment of package:%s version:%s to env:%s", deploymentTask.getApplicationName(), deploymentTask.getApplicationVersion(), deploymentTask.getEnvironment()));
		taskInfo.setState(deploymentTask.getState().name());
		taskInfo.setStartDate(deploymentTask.getStartDate());
		taskInfo.setCompletionDate(deploymentTask.getCompletionDate());
		taskInfo.setNrOfSteps(deploymentTask.getSteps().size());
		taskInfo.setCurrentStepNr(deploymentTask.getCurrentStepNr());
		taskInfo.setUser(deploymentTask.getOwner());
		taskInfo.setEnvironment(deploymentTask.getEnvironment());
		taskInfo.setApplication(deploymentTask.getApplicationName());
		taskInfo.setVersion(deploymentTask.getApplicationVersion());
        return taskInfo;
	}

    public RepositoryObject validationMessagesToDto(final RepositoryObjectEntity entity, final List<ValidationMessage> messages) {
        final RepositoryObject repositoryObject = entityToDto(entity);
        repositoryObject.setValidations(validationMessagesToDto(messages));
        return repositoryObject;
    }

    public List<Message> validationMessagesToDto(final List<ValidationMessage> messages) {
        final List<Message> list = Lists.newArrayList();
        for (ValidationMessage validationMessage : messages) {
            list.add(new Message(validationMessage.getFieldName(), validationMessage.getMessage()));
        }

        return list;
    }

	public FullTaskInfo fullArchivedTaskToDto(final DeploymentTaskInfo deploymentTaskInfo) {
		checkNotNull(deploymentTaskInfo, "deploymentTaskInfo");

		int stepNr = 1;
		FullTaskInfo fullTaskInfo = populateTaskInfo(deploymentTaskInfo, new FullTaskInfo());
        for (TaskStepInfo taskStepInfo : deploymentTaskInfo.getSteps()){
        	fullTaskInfo.addStep(taskStepInfoToDto(stepNr++, taskStepInfo));
        };
		return fullTaskInfo;
	}
}
