package com.xebialabs.deployit.core.rest.api;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.deployit.core.rest.api.DtoReader.ciDataToCiId;
import static com.xebialabs.deployit.core.rest.api.SearchParameterFactory.createSearchParam;
import static java.lang.String.format;

import java.text.SimpleDateFormat;
import java.util.List;
import java.util.Set;

import javax.ws.rs.core.Response;

import com.google.common.base.Function;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.core.api.resteasy.CustomMediaType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.core.api.ReportProxy;
import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemIds;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.api.reports.AggregatedDeploymentsKeyIndicator;
import com.xebialabs.deployit.core.rest.api.reports.DeploymentsForEnvironmentReport;
import com.xebialabs.deployit.core.rest.api.reports.DeploymentsKeyIndicator;
import com.xebialabs.deployit.core.rest.api.reports.DeploymentsStateBreakdownReport;
import com.xebialabs.deployit.core.rest.api.reports.ReportFilterType;
import com.xebialabs.deployit.core.rest.api.reports.TasksInDateRangeReport;
import com.xebialabs.deployit.core.rest.api.reports.widgets.WidgetRegistry;
import com.xebialabs.deployit.task.TaskArchive;

@Controller
public class ReportResource implements ReportProxy {

	@Autowired
	private TaskArchive taskArchive;

	@Override
	public Response deploymentsForEnvironment(final String environment, final Date date, String acceptedContentType) {
		checkNotNull(environment, "environment");
		checkNotNull(date, "date");
		checkNotNull(date.getCalendar(), "date");

		Report report = new DeploymentsForEnvironmentReport(taskArchive).report(environment, date);

		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
		return createResponse(report, format("deployments-%s-%s.csv", environment.replaceAll("/", "_"), dateFormat.format(date.getCalendar().getTime())),
		        acceptedContentType);
	}

    @Override
    public Response listEnvironments() {
        List<ConfigurationItemId> archivedEnvs = transform(taskArchive.getAllEnvironments(),new Function<String, ConfigurationItemId>() {
            @Override
            public ConfigurationItemId apply(String input) {
                return new ConfigurationItemId(input,"udm.Environment");
            }
        });

        return ResponseFactory.ok(new ConfigurationItemIds(archivedEnvs)).build();
    }

    @Override
	public Response tasksInDateRange(final Date begin, final Date end, String acceptedContentType) {
		checkNotNull(begin, "begin");
		checkNotNull(begin.getCalendar(), "begin");
		checkNotNull(end, "end");
		checkNotNull(end.getCalendar(), "end");

		Report report = new TasksInDateRangeReport(taskArchive).report(begin, end);
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
		return createResponse(report,
		        format("tasks-%s-%s.csv", dateFormat.format(begin.getCalendar().getTime()), dateFormat.format(end.getCalendar().getTime())),
		        acceptedContentType);
	}

	@Override
	public Response deploymentsStateBreakdown(String filterType, Date begin, Date end, String acceptedContentType, ConfigurationItemIds configurationItemIds) {
		checkNotNull(filterType, "filterType");
		checkNotNull(begin, "begin");
		checkNotNull(begin.getCalendar(), "begin");
		checkNotNull(end, "end");
		checkNotNull(end.getCalendar(), "end");
		checkNotNull(configurationItemIds.getConfigurationItemIds(), "configurationItemIds");

		ReportFilterType reportFilterType = ReportFilterType.valueOf(filterType.toUpperCase());
		Report report = new DeploymentsStateBreakdownReport(taskArchive).report(configurationItemIds.getConfigurationItemIds(), begin, end, reportFilterType);
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
		return createResponse(report,
		        format("deployments-by-state-%s-%s.csv", dateFormat.format(begin.getCalendar().getTime()), dateFormat.format(end.getCalendar().getTime())),
		        acceptedContentType);
	}

	@Override
	public Response deploymentsKeyIndicator(Date begin, Date end, String acceptedContentType) {
		checkNotNull(begin, "begin");
		checkNotNull(begin.getCalendar(), "begin");
		checkNotNull(end, "end");
		checkNotNull(end.getCalendar(), "end");

		Report report = new DeploymentsKeyIndicator(taskArchive).report(begin, end);
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
		return createResponse(
		        report,
		        format("deployments-key-indicator-%s-%s.csv", dateFormat.format(begin.getCalendar().getTime()), dateFormat.format(end.getCalendar().getTime())),
		        acceptedContentType);
	}

	@Override
	public Response deploymentsKeyIndicatorAggregatedByFilterType(String filterType, Date begin, Date end, String acceptedContentType,
	        ConfigurationItemIds configurationItemIds) {
		checkNotNull(filterType, "filterType");
		checkNotNull(begin, "begin");
		checkNotNull(begin.getCalendar(), "begin");
		checkNotNull(end, "end");
		checkNotNull(end.getCalendar(), "end");
		checkNotNull(configurationItemIds.getConfigurationItemIds(), "configurationItemIds");

		ReportFilterType reportFilterType = ReportFilterType.valueOf(filterType.toUpperCase());

		Report report = new AggregatedDeploymentsKeyIndicator(taskArchive).report(begin, end, reportFilterType, configurationItemIds.getConfigurationItemIds());
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
		return createResponse(
		        report,
		        format("aggregated-deployments-key-indicator-%s-%s-%s.csv", filterType, dateFormat.format(begin.getCalendar().getTime()),
		                dateFormat.format(end.getCalendar().getTime())), acceptedContentType);
	}

	@Override
	public Response getDashboardWidgetReport(String widget, Date begin, Date end) {
		checkNotNull(widget, "widget");
		checkNotNull(begin, "begin");
		checkNotNull(begin.getCalendar(), "begin");
		checkNotNull(end, "end");
		checkNotNull(end.getCalendar(), "end");
		checkArgument(begin.getCalendar().getTime().compareTo(end.getCalendar().getTime()) <= 0);
		return ResponseFactory.ok(WidgetRegistry.valueOf(WidgetRegistry.class, widget.toUpperCase()).getWidget(taskArchive).getReport(begin, end)).build();
	}

	private Response createResponse(Report report, String reportName, String acceptedContentType) {
		Response.ResponseBuilder responseBuilder = ResponseFactory.ok(report);
		if (CustomMediaType.TEXT_CSV.equals(acceptedContentType)) {
			responseBuilder.header("Content-Disposition", "attachment; filename=" + reportName);
		}
		return responseBuilder.build();
	}

    private static final Logger logger = LoggerFactory.getLogger(ReportResource.class);
}
