package com.xebialabs.deployit.core.rest.api.reports;

import static com.xebialabs.deployit.task.Task.State.CANCELLED;
import static com.xebialabs.deployit.task.Task.State.DONE;

import java.util.List;
import java.util.Map;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.DeploymentTaskInfo;
import com.xebialabs.deployit.task.TaskArchive;
import com.xebialabs.deployit.task.jcrarchive.JcrTaskArchive.TaskCallback;

public class DeploymentsStateBreakdownPerEnvironmentReport {

	private TaskArchive taskArchive;

	public DeploymentsStateBreakdownPerEnvironmentReport(TaskArchive taskArchive) {
		this.taskArchive = taskArchive;
	}

	public Report report(final List<String> environments, final Date startDate, final Date endDate) {
		ArchivedTaskSearchParameters searchTasks = new ArchivedTaskSearchParameters();
		searchTasks.createdBetween(startDate.getCalendar(), endDate.getCalendar());
		final Map<String, TaskStateData> tasksMap = Maps.newHashMap();
		for (String env : environments) {
			searchTasks.toEnvironment(env);
			taskArchive.searchTasksWithoutLoadingSteps(searchTasks, new TaskCallback() {
				public void doWithTask(DeploymentTaskInfo task) {
					createTasksMap(tasksMap, task);
				}
			});
		}

		for (String env : environments) {
			if (!tasksMap.containsKey(env)) {
				tasksMap.put(env, new TaskStateData());
			}
		}

		return generateReportDataFromTaskMap(tasksMap);
	}

	private Report generateReportDataFromTaskMap(Map<String, TaskStateData> tasksMap) {
		final Report report = new Report();
		for (Map.Entry<String, TaskStateData> entry : tasksMap.entrySet()) {
			final Report.ReportLine reportLine = report.addLine();
			reportLine.addValue("environmentName", entry.getKey());
			reportLine.addValue("noOfSuccessfulDeployments", entry.getValue().getNoOfSuccessfulDeployments().toString());
			reportLine.addValue("noOfFailedDeployments", entry.getValue().getNoOfFailedDeployments().toString());
		}
		return report;
	}

	private void createTasksMap(Map<String, TaskStateData> tasksMap, DeploymentTaskInfo task) {
		String targetEntityId = task.getEnvironment();

		if (task.getState().equals(DONE)) {
			if (!tasksMap.containsKey(targetEntityId)) {
				TaskStateData taskStateData = new TaskStateData();
				taskStateData.getNoOfSuccessfulDeployments().incrementAndGet();
				tasksMap.put(targetEntityId, taskStateData);
			} else {
				tasksMap.get(targetEntityId).getNoOfSuccessfulDeployments().incrementAndGet();
			}
		} else if (task.getState().equals(CANCELLED)) {
			if (!tasksMap.containsKey(targetEntityId)) {
				TaskStateData taskStateData = new TaskStateData();
				taskStateData.getNoOfAbortedDeployments().incrementAndGet();
				tasksMap.put(targetEntityId, taskStateData);
			} else {
				tasksMap.get(targetEntityId).getNoOfFailedDeployments().incrementAndGet();
			}
		}
	}
}
