package com.xebialabs.deployit.core.rest.api;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import javax.ws.rs.PathParam;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.StreamingOutput;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Controller;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.io.ByteStreams;
import com.google.common.io.Closeables;

import com.xebialabs.deployit.core.api.DownloadService;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.repository.WorkDir;

import static java.lang.String.format;

@Controller
public class DownloadResource implements DownloadService {

    private final Cache<String, Download> downloadCache = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES).build();

    @Override
    public Response download(@PathParam("downloadkey") final String downloadKey) {
        final Download download = downloadCache.getIfPresent(downloadKey);
        if (download == null) {
            throw new NotFoundException("The download with key [%s] was not found.", downloadKey);
        }

        downloadCache.invalidate(downloadKey);

        logger.info("Going to stream download of [{}]", download.file.getPath());

        StreamingOutput output = new StreamingOutput() {
            public void write(OutputStream output) throws IOException, WebApplicationException {
                InputStream is = null;
                try {
                    is = new FileInputStream(download.file);
                    ByteStreams.copy(is, output);
                } finally {
                    Closeables.closeQuietly(is);
                    download.workDir.delete();
                }
            }
        };

        return Response.ok(output, download.contentType)
                .header("Content-Disposition", format("attachment; filename=\"%s\"", download.file.getName()))
                .header("Content-Length", String.valueOf(download.file.length())).build();
    }

    public String register(File file, WorkDir workDir, String contentType) {
        Download download = new Download(file, workDir, contentType);

        logger.info("Registering download [{}] with key [{}]", download.file.getPath(), download.key);
        downloadCache.put(download.key, download);

        return download.key;
    }

    private static class Download {
        private final String key = UUID.randomUUID().toString();
        private final File file;
        private final WorkDir workDir;
        private final String contentType;

        Download(final File file, final WorkDir workDir, final String contentType) {
            this.file = file;
            this.workDir = workDir;
            this.contentType = contentType;
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(DownloadResource.class);
}
