package com.xebialabs.deployit.core.rest.api;

import java.util.List;

import org.springframework.stereotype.Component;

import com.google.common.base.Function;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;

import com.xebialabs.deployit.core.api.dto.RolePermissions;
import com.xebialabs.deployit.core.api.dto.RolePrincipals;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.ConfigurationItemData;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.permission.Permission;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static java.lang.String.format;

@Component
public class DtoReader {

    public static final Function<ConfigurationItem, ConfigurationItemId> ciToCiId = new Function<ConfigurationItem, ConfigurationItemId>() {
        public ConfigurationItemId apply(ConfigurationItem input) {
            return new ConfigurationItemId(input.getId(), input.getType());
        }
    };

    public static final Function<ConfigurationItemData, ConfigurationItemId> ciDataToCiId = new Function<ConfigurationItemData, ConfigurationItemId>() {
        public ConfigurationItemId apply(ConfigurationItemData input) {
            return new ConfigurationItemId(input.getId(), input.getType());
        }
    };

    public static final Function<ConfigurationItemId, ConfigurationItemData> ciIdToCiData = new Function<ConfigurationItemId, ConfigurationItemData>() {
        public ConfigurationItemData apply(ConfigurationItemId input) {
            return new ConfigurationItemData(input.getId(), input.getType());
        }
    };

    public List<Role> readRoleAssignments(List<RolePrincipals> assignments) {
        List<Role> roles = newArrayList();
        for (RolePrincipals roleAssignment : assignments) {
            com.xebialabs.deployit.engine.api.security.Role roleDto = roleAssignment.getRole();
            Role role = asRole(roleDto);
            List<String> principals = roleAssignment.getPrincipals();
            role.getPrincipalsAssigned().addAll(principals);
            roles.add(role);
        }

        return roles;
    }

    private static Role asRole(com.xebialabs.deployit.engine.api.security.Role roleDto) {
        return new Role(roleDto.getId(), roleDto.getName());
    }


    public Multimap<Role, Permission> readRolePermissions(List<RolePermissions> permissions, List<Role> roles) {
        Multimap<Role, Permission> multimap = HashMultimap.create();
        for (RolePermissions rolePermission : permissions) {
            Role role = asRole(rolePermission.getRole());
            if (role.getId() == null) {
                role = findRole(role.getName(), roles);
            }
            List<Permission> permissionList = transform(rolePermission.getPermissions(), new Function<String, Permission>() {
                public Permission apply(String input) {
                    return Permission.find(input);
                }
            });
            multimap.putAll(role, permissionList);
        }
        return multimap;
    }

    private static Role findRole(String name, List<Role> roles) {
        for (Role role : roles) {
            if (role.getName().equals(name)) {
                return role;
            }
        }
        throw new IllegalArgumentException(format("Couldn't find role with name [%s]", name));
    }
}
