package com.xebialabs.deployit.core.rest.api;

import java.io.File;
import java.nio.charset.Charset;
import java.util.Collections;
import java.util.List;

import javax.ws.rs.HeaderParam;
import javax.ws.rs.core.Response;

import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.google.common.base.Function;
import com.google.common.io.Files;

import com.xebialabs.deployit.core.api.InternalReportProxy;
import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.core.rest.api.reports.AggregatedDeploymentsKeyIndicator;
import com.xebialabs.deployit.core.rest.api.reports.DeploymentsForEnvironmentReport;
import com.xebialabs.deployit.core.rest.api.reports.DeploymentsKeyIndicator;
import com.xebialabs.deployit.core.rest.api.reports.DeploymentsStateBreakdownReport;
import com.xebialabs.deployit.core.rest.api.reports.TaskReport;
import com.xebialabs.deployit.core.rest.api.reports.widgets.WidgetRegistry;
import com.xebialabs.deployit.core.rest.resteasy.Workdir;
import com.xebialabs.deployit.core.rest.resteasy.WorkdirHolder;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.task.FilterType;
import com.xebialabs.deployit.task.archive.JcrTaskArchive;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.deployit.core.rest.api.reports.ReportUtils.toCsv;
import static java.lang.String.format;

@Controller
public class ReportResource implements InternalReportProxy {

    @Autowired
    private JcrTaskArchive taskArchive;

    @Autowired
    private DownloadResource downloadResource;

    private static final DateTimeFormatter DATE_FORMAT = DateTimeFormat.forPattern("yyyyMMdd");

    @Override
    public Report deploymentsForEnvironment(final String environment, final Date date, String acceptedContentType) {
        checkNotNull(environment, "environment");
        checkNotNull(date, "date");
        checkNotNull(date.getCalendar(), "date");

        DeploymentsForEnvironmentReport report = new DeploymentsForEnvironmentReport(taskArchive);
        report.getSearchParameters().forEnvironment(environment).createdOnOrBefore(date.asDateTime());

        return report.report();
    }

    @Workdir
    @Override
    public String downloadDeploymentsForEnvironment(final String environment, final Date date, @HeaderParam("Accept") final String acceptedContentType) {
        Report report = deploymentsForEnvironment(environment, date, acceptedContentType);
        String fileName = format("deployments-%s-%s.csv", environment.replaceAll("/", "_"), DATE_FORMAT.print(date.asDateTime()));

        return createDownloadToken(report, fileName);
    }

    @Override
    public Report getTaskReport(Date begin, Date end, String acceptedContentType, String filterType, List<ConfigurationItemId> configurationItemIds) {
        checkNotNull(begin, "begin");
        checkNotNull(end, "end");

        TaskReport report = new TaskReport(taskArchive);
        report.getSearchParameters().createdBetween(begin.asDateTime(), end.asDateTime());
        if (filterType != null) {
            FilterType filter = FilterType.valueOf(filterType.toUpperCase());
            if (filter != FilterType.NONE) {
                report.getSearchParameters().forFilterType(filter, configurationItemIds);
            }
        }

        return report.report();
    }

    @Workdir
    @Override
    public String downloadTaskReport(Date begin, Date end, String acceptedContentType, String filterType, List<ConfigurationItemId> configurationItemIds) {
        Report report = getTaskReport(begin, end, acceptedContentType, filterType, configurationItemIds);
        String fileName = format("tasks-%s-%s.csv", DATE_FORMAT.print(begin.asDateTime()), DATE_FORMAT.print(end.asDateTime()));
        return createDownloadToken(report, fileName);
    }

    @Override
    public Report deploymentsStateBreakdown(String filterType, Date begin, Date end, String acceptedContentType, List<ConfigurationItemId> configurationItemIds) {
        checkNotNull(filterType, "filterType");
        checkNotNull(begin, "begin");
        checkNotNull(begin.getCalendar(), "begin");
        checkNotNull(end, "end");
        checkNotNull(end.getCalendar(), "end");
        checkNotNull(configurationItemIds, "configurationItemIds");

        DeploymentsStateBreakdownReport report = new DeploymentsStateBreakdownReport(taskArchive);
        report.getSearchParameters().createdBetween(begin.asDateTime(), end.asDateTime());

        FilterType reportFilterType = FilterType.valueOf(filterType.toUpperCase());
        return report.report(reportFilterType, configurationItemIds);
    }

    @Workdir
    @Override
    public String downloadDeploymentsStateBreakdown(final String filterType, final Date begin, final Date end, @HeaderParam("Accept") final String acceptedContentType, final List<ConfigurationItemId> configurationItemIds) {
        Report report = deploymentsStateBreakdown(filterType, begin, end, acceptedContentType, configurationItemIds);
        String fileName = format("deployments-by-state-%s-%s.csv", DATE_FORMAT.print(begin.asDateTime()), DATE_FORMAT.print(end.asDateTime()));

        return createDownloadToken(report, fileName);
    }

    @Override
    public Report deploymentsKeyIndicator(Date begin, Date end, String acceptedContentType, String filterType, List<ConfigurationItemId> configurationItemIds) {
        checkNotNull(begin, "begin");
        checkNotNull(begin.getCalendar(), "begin");
        checkNotNull(end, "end");
        checkNotNull(end.getCalendar(), "end");

        DeploymentsKeyIndicator report = new DeploymentsKeyIndicator(taskArchive);
        report.getSearchParameters().createdBetween(begin.asDateTime(), end.asDateTime());
        if (filterType != null) {
            FilterType filter = FilterType.valueOf(filterType.toUpperCase());
            if (filter != FilterType.NONE) {
                report.getSearchParameters().forFilterType(filter, configurationItemIds);
            }
        }

        return report.report();
    }

    @Override
    public Report deploymentsKeyIndicatorAggregatedByFilterType(String filterType, Date begin, Date end, String acceptedContentType,
                                                                List<ConfigurationItemId> configurationItemIds) {
        checkNotNull(filterType, "filterType");
        checkNotNull(begin, "begin");
        checkNotNull(begin.getCalendar(), "begin");
        checkNotNull(end, "end");
        checkNotNull(end.getCalendar(), "end");
        checkNotNull(configurationItemIds, "configurationItemIds");

        FilterType reportFilterType = FilterType.valueOf(filterType.toUpperCase());

        AggregatedDeploymentsKeyIndicator report = new AggregatedDeploymentsKeyIndicator(taskArchive);
        report.getSearchParameters().createdBetween(new DateTime(begin.getCalendar()), new DateTime(end.getCalendar()));

        return report.report(reportFilterType, configurationItemIds);
    }

    @Workdir
    @Override
    public String downloadDeploymentsKeyIndicator(String filterType, Date begin, Date end, String acceptedContentType, List<ConfigurationItemId> cis) {
        Report report = deploymentsKeyIndicator(begin, end, acceptedContentType, filterType, cis);
        String fileName = format("deployments-key-indicator-%s-%s.csv", DATE_FORMAT.print(begin.asDateTime()), DATE_FORMAT.print(end.asDateTime()));

        return createDownloadToken(report, fileName);
    }

    @Workdir
    @Override
    public String downloadDeploymentsKeyIndicatorAggregatedByFilterType(final String filterType, final Date begin, final Date end, @HeaderParam("Accept") final String acceptedContentType, final List<ConfigurationItemId> configurationItemIds) {
        Report report = deploymentsKeyIndicatorAggregatedByFilterType(filterType, begin, end, acceptedContentType, configurationItemIds);
        String fileName = format("aggregated-deployments-key-indicator-%s-%s-%s.csv", filterType, DATE_FORMAT.print(begin.asDateTime()), DATE_FORMAT.print(end.asDateTime()));

        return createDownloadToken(report, fileName);
    }

    @Override
    public Response getDashboardWidgetReport(String widget, Date begin, Date end) {
        checkNotNull(widget, "widget");
        checkNotNull(begin, "begin");
        checkNotNull(begin.getCalendar(), "begin");
        checkNotNull(end, "end");
        checkNotNull(end.getCalendar(), "end");
        checkArgument(begin.getCalendar().getTime().compareTo(end.getCalendar().getTime()) <= 0);

        return Response.ok(WidgetRegistry.valueOf(WidgetRegistry.class, widget.toUpperCase()).getWidget(taskArchive).getReport(begin.asDateTime(), end.asDateTime())).build();
    }

    @Override
    public List<ConfigurationItemId> listEnvironments() {
        List<String> environments = taskArchive.getAllEnvironments();
        Collections.sort(environments);

        return newArrayList(transform(environments, toConfigurationItemId("udm.Environment")));
    }

    private static Function<String, ConfigurationItemId> toConfigurationItemId(final String type) {
        return new Function<String, ConfigurationItemId>() {
            @Override
            public ConfigurationItemId apply(String input) {
                return new ConfigurationItemId(input,Type.valueOf(type));
            }
        };
    }

    private String createDownloadToken(Report report, String fileName) {
        try {
            File to = new File(WorkdirHolder.get().getPath(), fileName);
            Files.write(toCsv(report), to, Charset.defaultCharset());
            return downloadResource.register(to, WorkdirHolder.get(), "text/csv");
        } catch (Exception re) {
            LoggerFactory.getLogger(getClass()).error("Oops!", re);
            WorkdirHolder.get().delete();
            throw new IllegalStateException(re);
        }
    }
}
