package com.xebialabs.deployit.core.rest.api;

import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.google.common.base.Function;
import com.google.common.collect.Multimap;

import com.xebialabs.deployit.core.api.InternalSecurityProxy;
import com.xebialabs.deployit.core.api.dto.RolePermissions;
import com.xebialabs.deployit.core.api.dto.RolePrincipals;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.spi.event.RolePermissionsChangedEvent;
import com.xebialabs.deployit.engine.spi.event.RolePrincipalsChangedEvent;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.security.PermissionEditor;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.permission.Permission;

import static com.google.common.collect.Lists.transform;
import static com.xebialabs.deployit.security.permission.Permission.EDIT_SECURITY;


@Controller
public class SecurityResource extends AbstractSecuredResource implements InternalSecurityProxy {

    @Autowired
    private RoleService roleService;

    @Autowired
    private PermissionEditor permissionEditor;

    @Autowired
    private DtoWriter dtoWriter;

    @Autowired
    private DtoReader dtoReader;

    @Override
    public List<com.xebialabs.deployit.engine.api.security.Role> list() {
        return dtoWriter.writeRoles(roleService.getRoles());
    }

    @Override
    public List<RolePermissions> readRolePermissions(String id) {
        checkPermission(EDIT_SECURITY);

        Multimap<Role, Permission> rolePermissionMultimap = permissionEditor.readPermissions(id);
        List<RolePermissions> rolePermissions = dtoWriter.writeRolePermissions(rolePermissionMultimap);
        return rolePermissions;
    }

    @Override
    public void writeRolePermissions(String id, List<RolePermissions> permissions) {
        checkPermission(EDIT_SECURITY);

        List<Role> roles = roleService.getRoles();
        Multimap<Role, Permission> multimap = dtoReader.readRolePermissions(permissions, roles);
        permissionEditor.editPermissions(id, multimap);

        fireRolePermissionsChangedEvent(id, permissions);
    }

    private static void fireRolePermissionsChangedEvent(String id, List<RolePermissions> permissions) {
        EventBusHolder.publish(new RolePermissionsChangedEvent(id, transform(permissions, new Function<RolePermissions, String>() {
            @Override
            public String apply(RolePermissions input) {
                return input.getRole().getName() + " => " + input.getPermissions();
            }
        })));
    }

    @Override
    public List<RolePrincipals> readRolePrincipals() {
        checkPermission(EDIT_SECURITY);
        List<Role> multimap = roleService.readRoleAssignments();
        logger.debug("RoleService delivered roles: {}", multimap);
        List<RolePrincipals> assignments = dtoWriter.writeRoleAssignments(multimap);
        logger.debug("Going to return the following roles: {}", assignments);
        return assignments;
    }

    @Override
    public void writeRolePrincipals(List<RolePrincipals> rolePrincipals) {
        checkPermission(EDIT_SECURITY);
        logger.debug("Got the following roles: {}", rolePrincipals);
        List<Role> roles = dtoReader.readRoleAssignments(rolePrincipals);
        logger.debug("Going to write the following role assignments: {}", roles);
        roleService.writeRoleAssignments(roles);

        fireRolePrincipalsChangedEvent(rolePrincipals);
    }

    private static void fireRolePrincipalsChangedEvent(List<RolePrincipals> rolePrincipals) {
        EventBusHolder.publish(new RolePrincipalsChangedEvent(transform(rolePrincipals, new Function<RolePrincipals, String>() {
            @Override
            public String apply(RolePrincipals input) {
                return input.getRole().getName() + " => " + input.getPrincipals();
            }
        })));
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
    public static class UnknownPermissionException extends DeployitException {
        public UnknownPermissionException(String permission) {
            super("Permission %s does not exist.", permission);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(SecurityResource.class);
}




