package com.xebialabs.deployit.core.rest.api;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.security.User;
import com.xebialabs.deployit.engine.spi.event.UserCreatedEvent;
import com.xebialabs.deployit.engine.spi.event.UserDeletedEvent;
import com.xebialabs.deployit.engine.spi.event.UserPasswordChangedEvent;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.security.JcrUser;
import com.xebialabs.deployit.security.UserService;
import com.xebialabs.deployit.security.permission.Permission;

@Controller
public class UserResource extends AbstractSecuredResource implements com.xebialabs.deployit.engine.api.UserService {

    @Autowired
    UserService userService;

    @Override
    public User create(final String username, final User user) {
        checkPermission(Permission.EDIT_SECURITY);
        sanityCheckUser(username, user);

        logger.debug("Creating user {}", username);
        userService.create(username, user.getPassword());
        JcrUser u = userService.read(username);
        logger.debug("Created user {}", u.getUsername());

        EventBusHolder.publish(new UserCreatedEvent(username));

        return new User(u.getUsername(), u.isAdmin());
    }

    @Override
    public User read(final String username) {
        checkPermission(Permission.EDIT_SECURITY);

        JcrUser u = userService.read(username);

        return new User(u.getUsername(), u.isAdmin());
    }

    @Override
    public void modifyPassword(final String username, final User user) {
        checkPermission(Permission.EDIT_SECURITY);
        sanityCheckUser(username, user);

        userService.modifyPassword(username, user.getPassword());
        
        EventBusHolder.publish(new UserPasswordChangedEvent(username));
    }

    @Override
    public void delete(final String username) {
        checkPermission(Permission.EDIT_SECURITY);

        userService.delete(username);
        
        EventBusHolder.publish(new UserDeletedEvent(username));
    }

    private static void sanityCheckUser(final String username, final User user) {
        if (user.getUsername() != null && !user.getUsername().equals(username)) {
            throw new IllegalArgumentException("Username in URL is " + username + " but username in POST data is " + user.getUsername());
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(UserResource.class);
}
