package com.xebialabs.deployit.core.rest.api.reports;

import java.util.List;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.collect.MapMaker;

import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.FilterType;
import com.xebialabs.deployit.task.archive.ArchivedTask;
import com.xebialabs.deployit.task.archive.JcrTaskArchive;
import com.xebialabs.deployit.task.archive.JcrTaskArchive.TaskCallback;

public class AggregatedDeploymentsKeyIndicator extends TaskArchiveReport {

    public AggregatedDeploymentsKeyIndicator(JcrTaskArchive taskArchive) {
        super(taskArchive);
    }

    @SuppressWarnings("deprecation")
    public Report report(final FilterType filterType, List<ConfigurationItemId> filterCriteria) {
        final Map<String, TaskStateData> tasks = new MapMaker().makeComputingMap(new Function<String, TaskStateData>() {
            public TaskStateData apply(String input) {
                return new TaskStateData();
            }
        });

        for (ConfigurationItemId appOrEnvName : filterCriteria) {
            ArchivedTaskSearchParameters searchTasks = cloneSearchParameters().forFilterType(filterType, appOrEnvName);

            taskArchive.searchTasksWithoutLoadingSteps(searchTasks, new TaskCallback() {
                public void doWithTask(ArchivedTask task) {
                    String entityId = getEntityId(task, filterType);
                    tasks.get(entityId).addTaskAndDuration(task);
                }
            });
        }

        return generateReport(tasks);
    }

    private static Report generateReport(Map<String, TaskStateData> tasks) {
        final Report report = new Report();
        for (Map.Entry<String, TaskStateData> entry : tasks.entrySet()) {
            final Report.ReportLine line = report.addLine();
            TaskStateData task = entry.getValue();
            line.addValue("entityName", entry.getKey());
            line.addValue("noOfSuccessfulDeployments", task.getNoOfSuccessfulDeployments().toString());
            line.addValue("noOfFailedDeployments", task.getNoOfFailedDeployments().toString());
            line.addValue("noOfAbortedDeployments", task.getNoOfAbortedDeployments().toString());
            line.addValue("noOfRollbacks", task.getNoOfRollbacks().toString());
            line.addValue("averageDuration", DeploymentsKeyIndicator.getAverageDeploymentTime(task));
        }
        return report;
    }

    private static String getEntityId(TaskState task, FilterType filterType) {
        switch (filterType) {
        case APPLICATION:
            return task.getMetadata().get("application");
        case ENVIRONMENT:
            return task.getMetadata().get("environment_id");
        }
        throw new IllegalArgumentException("Unknown filter type: " + filterType);
    }
}
