package com.xebialabs.deployit.core.rest.api.reports;

import java.util.List;
import java.util.Map;

import org.joda.time.DateTime;

import com.google.common.collect.Maps;

import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.archive.ArchivedTask;
import com.xebialabs.deployit.task.archive.JcrTaskArchive;
import com.xebialabs.deployit.task.archive.JcrTaskArchive.TaskCallback;

public class DeploymentsStateBreakdownPerEnvironmentReport extends TaskArchiveReport {

    public DeploymentsStateBreakdownPerEnvironmentReport(JcrTaskArchive taskArchive) {
        super(taskArchive);
    }

    public Report report(final List<String> environments, final DateTime startDate, final DateTime endDate) {
        getSearchParameters().createdBetween(startDate, endDate);

        final Map<String, TaskStateData> tasks = Maps.newHashMap();
        for (String env : environments) {
            ArchivedTaskSearchParameters searchTasks = cloneSearchParameters().forEnvironment(env);
            taskArchive.searchTasksWithoutLoadingSteps(searchTasks, new TaskCallback() {
                public void doWithTask(ArchivedTask task) {
                    addTask(tasks, task);
                }
            });
        }

        for (String env : environments) {
            if (!tasks.containsKey(env)) {
                tasks.put(env, new TaskStateData());
            }
        }

        return generateReport(tasks);
    }

    private static Report generateReport(Map<String, TaskStateData> tasks) {
        final Report report = new Report();
        for (Map.Entry<String, TaskStateData> entry : tasks.entrySet()) {
            final Report.ReportLine line = report.addLine();
            line.addValue("environmentName", entry.getKey());
            line.addValue("noOfSuccessfulDeployments", entry.getValue().getNoOfSuccessfulDeployments().toString());
            line.addValue("noOfFailedDeployments", entry.getValue().getNoOfFailedDeployments().toString());
        }
        return report;
    }

    private static void addTask(Map<String, TaskStateData> tasks, TaskState task) {
        String targetEntityId = task.getMetadata().get("environment");

        if (task.getState().equals(TaskExecutionState.DONE)) {
            if (!tasks.containsKey(targetEntityId)) {
                TaskStateData taskStateData = new TaskStateData();
                taskStateData.getNoOfSuccessfulDeployments().incrementAndGet();
                tasks.put(targetEntityId, taskStateData);
            } else {
                tasks.get(targetEntityId).getNoOfSuccessfulDeployments().incrementAndGet();
            }
        } else if (task.getState().equals(TaskExecutionState.CANCELLED)) {
            if (!tasks.containsKey(targetEntityId)) {
                TaskStateData taskStateData = new TaskStateData();
                taskStateData.getNoOfAbortedDeployments().incrementAndGet();
                tasks.put(targetEntityId, taskStateData);
            } else {
                tasks.get(targetEntityId).getNoOfFailedDeployments().incrementAndGet();
            }
        }
    }
}
