package com.xebialabs.deployit.core.rest.api.reports;

import java.util.List;
import java.util.Map;

import com.google.common.collect.Maps;

import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.FilterType;
import com.xebialabs.deployit.task.archive.ArchivedTask;
import com.xebialabs.deployit.task.archive.JcrTaskArchive;
import com.xebialabs.deployit.task.archive.JcrTaskArchive.TaskCallback;

import static com.xebialabs.deployit.engine.api.execution.TaskExecutionState.DONE;

public class DeploymentsStateBreakdownReport extends TaskArchiveReport {

    public DeploymentsStateBreakdownReport(JcrTaskArchive taskArchive) {
        super(taskArchive);
    }

    public Report report(final FilterType filterType, final List<ConfigurationItemId> filterCriteria) {

        final Map<String, TaskStateData> tasks = Maps.newHashMap();
        for (ConfigurationItemId appOrEnvName : filterCriteria) {
            ArchivedTaskSearchParameters searchParameters = cloneSearchParameters();
            searchParameters.forFilterType(filterType, appOrEnvName);

            taskArchive.searchTasksWithoutLoadingSteps(searchParameters, new TaskCallback() {
                public void doWithTask(ArchivedTask task) {
                    incrementTaskCount(tasks, task, filterType);
                }
            });
        }

        for (ConfigurationItemId entityId : filterCriteria) {
            if (!tasks.containsKey(entityId.getId())) {
                tasks.put(entityId.getId(), new TaskStateData());
            }
        }
        return generateReportData(tasks);
    }

    private static Report generateReportData(Map<String, TaskStateData> tasks) {
        final Report report = new Report();
        for (Map.Entry<String, TaskStateData> entry : tasks.entrySet()) {
            final Report.ReportLine reportLine = report.addLine();
            reportLine.addValue("entityName", entry.getKey());
            reportLine.addValue("noOfSuccessfulDeployments", entry.getValue().getNoOfSuccessfulDeployments().toString());
        }
        return report;
    }

    private static void incrementTaskCount(Map<String, TaskStateData> tasks, TaskState task, FilterType filterType) {
        String entityId = getEntityId(task, filterType);
        if (task.getState() ==DONE) {
            TaskStateData taskStateData = tasks.get(entityId);
            if (taskStateData == null) {
                taskStateData = new TaskStateData();
                tasks.put(entityId, taskStateData);
            }
            taskStateData.getNoOfSuccessfulDeployments().incrementAndGet();
        }
    }

    private static String getEntityId(TaskState task, FilterType filterType) {
        String entityId = "";
        switch (filterType) {
        case APPLICATION:
            entityId = task.getMetadata().get("application");
            break;
        case ENVIRONMENT:
            entityId = task.getMetadata().get("environment_id");
            break;
        default:
            throw new UnsupportedOperationException("Report filter type, " + filterType + " is not supported");
        }

        return entityId;
    }

}
