package com.xebialabs.deployit.core.rest.api.reports.widgets;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;
import java.util.Map;

import org.joda.time.DateTime;

import com.google.common.base.Function;

import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.dto.Report.ReportLine;
import com.xebialabs.deployit.core.rest.api.reports.TaskStateData;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.archive.ArchivedTask;
import com.xebialabs.deployit.task.archive.JcrTaskArchive;
import com.xebialabs.deployit.task.archive.JcrTaskArchive.TaskCallback;

/*********************
 * Throughput widget
 */
public class DeploymentTrendsWidget extends DashboardWidgetBase {

    public DeploymentTrendsWidget(final JcrTaskArchive taskArchive) {
        super(taskArchive);
    }

    @Override
    public final Report getReport(DateTime begin, DateTime end) {
        Map<String, TaskStateData> taskStates = groupTasksByStateAndDate(begin, end);
        return generateReportData(taskStates);
    }

    private static Report generateReportData(Map<String, TaskStateData> taskStates) {
        final Report report = new Report();
        DecimalFormat decimalFormat = new DecimalFormat("#.##",DecimalFormatSymbols.getInstance(Locale.ENGLISH));
        for (String month : taskStates.keySet()) {
            TaskStateData stateData = taskStates.get(month);
            ReportLine line = report.addLine();
            line.addValue("month", month);
            line.addValue("noOfSuccessfulDeployments", stateData.getNoOfSuccessfulDeployments().toString());
            line.addValue("noOfAbortedDeployments", stateData.getNoOfAbortedDeployments().toString());
            line.addValue("noOfFailedDeployments", stateData.getNoOfFailedDeployments().toString());
            line.addValue("noOfRollbacks", stateData.getNoOfRollbacks().toString());
            long totalTasks = stateData.getTotalTasks();
            line.addValue("percentageSuccessfulDeployments", decimalFormat.format(getPercentage(stateData.getNoOfSuccessfulDeployments().get(), totalTasks)));
        }
        return report;
    }

    /**
     * Groups only completed tasks by state. Ignores the middle states like EXECUTING or PENDING.
     *
     */
    private Map<String, TaskStateData> groupTasksByStateAndDate(final DateTime begin, DateTime end) {
        final Map<String, TaskStateData> stateDataMap = initializeMap(begin, end, new Function<String, TaskStateData>() {
            public TaskStateData apply(String input) {
                return new TaskStateData();
            }
        });

        ArchivedTaskSearchParameters params = getSearchParameters().createdBetween(begin, end);

        taskArchive.searchTasksWithoutLoadingSteps(params, new TaskCallback() {
            public void doWithTask(ArchivedTask task) {
                String month = monthFormat.print(task.getCompletionDate());
                TaskStateData dataPerMonth = stateDataMap.get(month);
                if(dataPerMonth != null) {
                    dataPerMonth.addTask(task);
                }
            }
        });
        return stateDataMap;
    }
}
