package com.xebialabs.deployit.core.rest.api.reports.widgets;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;

import org.joda.time.DateTime;

import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.dto.Report.ReportLine;
import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.archive.ArchivedTask;
import com.xebialabs.deployit.task.archive.JcrTaskArchive;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.core.rest.api.reports.ReportUtils.duration;
import static com.xebialabs.deployit.task.TaskType.*;

/*********************
 * Deployment duration over time widget
 */
public class DeploymentsDurationFrequencyWidget extends DashboardWidgetBase {

    private static final String INTERVAL = "interval";
    private static final String DEPLOYMENT_PERCENT = "deploymentPercent";
    private static final String NUM_OF_DEPLOYMENT = "numOfDeployment";

    private final DecimalFormat format = new DecimalFormat("#.##", DecimalFormatSymbols.getInstance(Locale.ENGLISH));

    public DeploymentsDurationFrequencyWidget(JcrTaskArchive taskArchive) {
        super(taskArchive);
    }

    @Override
    public Report getReport(DateTime begin, DateTime end) {
        final ArchivedTaskSearchParameters sp = getSearchParameters().createdBetween(begin, end).thatCompleted().thatAreOfType(EnumSet.of(INITIAL, UNDEPLOY, UPGRADE));
        final Collection<ArchivedTask> foundTasks = taskArchive.searchTasksWithoutLoadingSteps(sp);
        if (foundTasks.size() == 0) {
            return new Report();
        }
        for (ArchivedTask foundTask : foundTasks) {
            System.out.println("M: " + foundTask.getMetadata());
        }
        return generateReport(foundTasks);
    }

    private Report generateReport(Collection<ArchivedTask> foundTasks) {
        final long mean = mean(foundTasks);
        final long standardDeviation = standardDeviation(mean, foundTasks);
        final List<Long> clusterLimits = clusterLimits(mean, standardDeviation);
        return createDetails(foundTasks, clusterLimits);
    }

    private static long mean(Collection<ArchivedTask> tasks) {
        long total = 0;
        for (TaskState taskState : tasks) {
            total += duration(taskState);
        }
        return total / tasks.size();// Ignoring the fractions
    }

    private static long standardDeviation(long mean, Collection<ArchivedTask> tasks) {
        if (tasks.size() > 1) {
            long total = 0;
            for (TaskState taskState : tasks) {
                final long distance = duration(taskState) - mean;
                total += distance * distance;
            }
            return (long) Math.sqrt(total / (tasks.size() - 1));// Ignore the fractions
        } else
            return 0;
    }

    private static List<Long> clusterLimits(long mean, long standardDeviation) {
        List<Long> clusterLimits = newArrayList();
        if ((standardDeviation > 0) && (mean - standardDeviation - standardDeviation > 0)) {
            clusterLimits.add(mean - standardDeviation - standardDeviation);
        }
        if ((standardDeviation > 0) && (mean - standardDeviation > 0)) {
            clusterLimits.add(mean - standardDeviation);
        }
        clusterLimits.add(mean);
        if (standardDeviation > 0) {
            clusterLimits.add(mean + standardDeviation);
            clusterLimits.add(mean + standardDeviation + standardDeviation);
        }
        return clusterLimits;
    }

    private Report createDetails(Collection<ArchivedTask> tasks, final List<Long> clusterLimits) {
        final Report report = new Report();
        final int[] clusters = cluster(tasks, clusterLimits);
        int totalDeployments = tasks.size();
        if (totalDeployments > 1) {
            ReportLine firstLine = report.addLine();
            firstLine.addValue(INTERVAL, "<" + formatToMinsAndSecs(clusterLimits.get(0)));
            firstLine.addValue(DEPLOYMENT_PERCENT, format.format(getPercentage(clusters[0], totalDeployments)));
            firstLine.addValue(NUM_OF_DEPLOYMENT, format.format(clusters[0]));
        }

        for (int i = 1; i < clusterLimits.size(); i++) {
            ReportLine line = report.addLine();
            line.addValue(INTERVAL, formatToMinsAndSecs(clusterLimits.get(i - 1)) + " - " + formatToMinsAndSecs(clusterLimits.get(i)));
            line.addValue(DEPLOYMENT_PERCENT, format.format(getPercentage(clusters[i], totalDeployments)));
            line.addValue(NUM_OF_DEPLOYMENT, format.format(clusters[i]));
        }

        ReportLine lastLine = report.addLine();
        lastLine.addValue(INTERVAL, ">" + formatToMinsAndSecs(clusterLimits.get(clusterLimits.size() - 1)));
        lastLine.addValue(DEPLOYMENT_PERCENT, format.format(getPercentage(clusters[clusters.length - 1], totalDeployments)));
        lastLine.addValue(NUM_OF_DEPLOYMENT, format.format(clusters[clusters.length - 1]));
        return report;
    }

    private static int[] cluster(Collection<ArchivedTask> tasks, List<Long> clusterLimits) {
        int[] cluster = new int[clusterLimits.size() + 1];

        for (TaskState taskState : tasks) {
            boolean aboveMaxLimit = true;
            for (int i = 0; i < clusterLimits.size(); i++) {
                Long limit = clusterLimits.get(i);
                if (duration(taskState) < limit) {
                    cluster[i]++;
                    aboveMaxLimit = false;
                    break;
                }
            }
            if (aboveMaxLimit)
                cluster[clusterLimits.size()]++;
        }
        return cluster;
    }
}
