package com.xebialabs.deployit.core.rest.xml;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;

import com.xebialabs.deployit.core.*;
import com.xebialabs.deployit.core.StringValue;
import com.xebialabs.deployit.core.rest.resteasy.WorkdirHolder;
import com.xebialabs.deployit.core.xml.ConfigurationItemWriter;
import com.xebialabs.deployit.engine.xml.XStreamProvider;
import com.xebialabs.deployit.io.DerivedArtifactFile;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.EncryptedDictionary;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.StringValueConverter;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_STRING;

@Component
@XStreamProvider(tagName = "configuration-item", readable = ConfigurationItem.class)
public class ConfigurationItemReaderWriter extends ConfigurationItemWriter {

    private final RepositoryService repository;
    private final StringValueConverter converter;

    @Autowired
    public ConfigurationItemReaderWriter(RepositoryService repository) {
        this.repository = repository;
        this.converter = new StringValueConverter(passwordEncrypter);
    }

    static final ThreadLocal<Map<String, ConfigurationItem>> DESERIALIZATION_CONTEXT = new ThreadLocal<Map<String, ConfigurationItem>>() {
        @Override
        protected Map<String, ConfigurationItem> initialValue() {
            return newHashMap();
        }
    };

    static final ThreadLocal<List<Reference>> DESERIALIZATION_REFERENCES = new ThreadLocal<List<Reference>>() {
        @Override
        protected List<Reference> initialValue() {
            return newArrayList();
        }
    };

    @Override
    protected void readStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
        if (propertyDescriptor.isPassword()) {
            propertyDescriptor.set(configurationItem, passwordEncrypter.ensureDecrypted(reader.getValue()));
        } else {
            super.readStringProperty(configurationItem, propertyDescriptor, reader);
        }
    }

    @Override
    protected void readCollectionOfStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
        Collection<StringValue> strings = (propertyDescriptor.getKind() == SET_OF_STRING ? Sets.<StringValue> newHashSet() : Lists
            .<StringValue> newArrayList());
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            strings.add(converter.convert(reader.getValue()));
            reader.moveUp();
        }

        @SuppressWarnings("rawtypes")
        AbstractStringView view = (propertyDescriptor.getKind() == SET_OF_STRING ? new SetOfStringView((Set<StringValue>) strings)
            : new ListOfStringView((List<StringValue>) strings));
        propertyDescriptor.set(configurationItem, view);
    }

    @Override
    protected void readMapStringStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
        Map<String, StringValue> map = Maps.newHashMap();
        boolean encrypt = configurationItem.getType().equals(Type.valueOf(EncryptedDictionary.class));
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            map.put(reader.getAttribute("key"), converter.convert(reader.getValue(), encrypt));
            reader.moveUp();
        }
        propertyDescriptor.set(configurationItem, new MapStringStringView(map));

    }

    @SuppressWarnings("rawtypes")
    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
        ConfigurationItem ci = (ConfigurationItem) super.unmarshal(reader, context);
        if (ci instanceof DerivedArtifact && WorkdirHolder.get() != null) {
            logger.trace("Setting DerivedFile on {}", ci.getId());
            ((DerivedArtifact) ci).setFile(DerivedArtifactFile.create((DerivedArtifact) ci));
        }
        DESERIALIZATION_CONTEXT.get().put(ci.getId(), ci);
        return ci;
    }

    @Override
    protected void readCiProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
        String id = reader.getAttribute("ref");
        DESERIALIZATION_REFERENCES.get().add(new ConfigurationItemReference(id, configurationItem, propertyDescriptor, repository));
    }

    @Override
    protected void readCollectionOfCiProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
        Collection<String> cis = null;
        if (propertyDescriptor.getKind() == LIST_OF_CI) {
            cis = newArrayList();
        } else {
            cis = newHashSet();
        }
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            cis.add(reader.getAttribute("ref"));
            reader.moveUp();
        }
        DESERIALIZATION_REFERENCES.get().add(new ConfigurationItemsReference(cis, configurationItem, propertyDescriptor, repository));
    }

    static void postProcess() {
        try {
            for (ConfigurationItemReaderWriter.Reference reference : ConfigurationItemReaderWriter.DESERIALIZATION_REFERENCES.get()) {
                reference.resolve();
            }
        } finally {
            clear();
        }
    }

    public static void clear() {
        DESERIALIZATION_REFERENCES.remove();
        DESERIALIZATION_CONTEXT.remove();
    }

    static interface Reference {
        void resolve();
    }

    private static class ConfigurationItemReference implements Reference {
        private String id;
        private ConfigurationItem owner;
        private PropertyDescriptor property;
        private RepositoryService repository;

        private ConfigurationItemReference(String id, ConfigurationItem owner, PropertyDescriptor property, RepositoryService repository) {
            this.id = id;
            this.owner = owner;
            this.property = property;
            this.repository = repository;
        }

        @Override
        public void resolve() {
            ConfigurationItem ci = null;
            if (DESERIALIZATION_CONTEXT.get().containsKey(id)) {
                ci = DESERIALIZATION_CONTEXT.get().get(id);
            } else {
                ci = repository.read(id, WorkdirHolder.get());
            }
            property.set(owner, ci);
        }
    }

    private static class ConfigurationItemsReference implements Reference {
        private Collection<String> ids;
        private ConfigurationItem owner;
        private PropertyDescriptor property;
        private RepositoryService repository;

        private ConfigurationItemsReference(Collection<String> ids, ConfigurationItem owner, PropertyDescriptor property, RepositoryService repository) {
            this.ids = ids;
            this.owner = owner;
            this.property = property;
            this.repository = repository;
        }

        @Override
        public void resolve() {
            Collection<ConfigurationItem> cis = null;
            if (property.getKind() == LIST_OF_CI) {
                cis = newArrayList();
            } else {
                cis = newHashSet();
            }

            for (String id : ids) {
                if (DESERIALIZATION_CONTEXT.get().containsKey(id)) {
                    cis.add(DESERIALIZATION_CONTEXT.get().get(id));
                } else {
                    cis.add(repository.read(id, WorkdirHolder.get()));
                }

            }
            property.set(owner, cis);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(ConfigurationItemReaderWriter.class);

}
