package com.xebialabs.deployit.service.importer.reader;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.engine.spi.artifact.resolution.ArtifactResolver;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Application;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Version;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.server.api.importer.ImportedPackage;
import com.xebialabs.deployit.server.api.importer.ImportingContext;
import com.xebialabs.deployit.server.api.importer.PackageInfo;
import com.xebialabs.deployit.service.importer.ImporterException;
import com.xebialabs.deployit.util.GuavaFiles;
import com.xebialabs.deployit.util.TFiles;
import com.xebialabs.overthere.local.LocalFile;
import com.xebialabs.xlplatform.artifact.resolution.ArtifactResolverRegistry;
import com.xebialabs.xltype.serialization.CiReference;
import de.schlichtherle.truezip.file.TArchiveDetector;
import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TVFS;
import de.schlichtherle.truezip.fs.FsSyncException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;
import static com.xebialabs.deployit.server.api.util.IdGenerator.generateId;
import static java.lang.String.format;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;

public class ManifestReaderDriver {
    static final String REFERENCES = "references";
    static final String READ_CIS = "readCis";
    static final String TEMPORARY_FILES = "temporaryFiles";

    private ManifestReader reader;
    private ImportSource source;

    public ManifestReaderDriver(ImportSource source, ManifestReader reader) {
        this.source = source;
        this.reader = reader;
    }

    public PackageInfo readPackageInfo(ImportingContext context) {
        PackageInfo packageInfo = new PackageInfo(source);
        reader.readPackageData(packageInfo, context);
        context.setAttribute(TEMPORARY_FILES, Lists.<TFile>newArrayList());
        return packageInfo;
    }

    public ImportedPackage readDeploymentPackage(PackageInfo packageInfo, ImportingContext context) {
        Application app = getDistribution(packageInfo, context);
        Version version = readVersion(packageInfo, app, context);
        return new ImportedPackage(packageInfo, app, version);
    }

    private Application getDistribution(PackageInfo packageInfo, ImportingContext context) {
        ManifestReader.ManifestCiReader ciReader = reader.versionReader();
        Type distributionType = ciReader.distributionType();
        Application app = distributionType.getDescriptor().newInstance(packageInfo.getApplicationId());
        getReadCis(context).put(packageInfo.getApplicationName(), app);
        return app;
    }

    private Version readVersion(PackageInfo packageInfo, Application app, ImportingContext context) {
        ManifestReader.ManifestCiReader ciReader = reader.versionReader();
        Descriptor versionDescriptor = ciReader.type().getDescriptor();

        Version v = versionDescriptor.newInstance(generateId(app.getId(), packageInfo.getApplicationVersion()));

        versionDescriptor
                .getPropertyDescriptors()
                .forEach(propertyDescriptor ->
                        readProperty(ciReader, propertyDescriptor, v, context, packageInfo));

        getReadCis(context).put(packageInfo.getApplicationVersion(), v);
        return v;
    }

    private String getPropertyNameInManifest(ManifestReader.ManifestCiReader ciReader, PropertyDescriptor pd) {
        if (ciReader.hasProperty(pd.getName())) {
            return pd.getName();
        } else {
            for (String alias : pd.getAliases()) {
                if (ciReader.hasProperty(alias)) {
                    return alias;
                }
            }
        }
        return null;
    }

    private void readProperty(ManifestReader.ManifestCiReader ciReader, PropertyDescriptor pd, ConfigurationItem ci, ImportingContext context, PackageInfo packageInfo) {
        String propNameInManifest = getPropertyNameInManifest(ciReader, pd);
        if (propNameInManifest == null) {
            return;
        }

        switch (pd.getKind()) {
            case BOOLEAN:
            case INTEGER:
            case STRING:
            case ENUM:
            case DATE:
                String val = ciReader.propertyAsString(propNameInManifest);
                if (!isBlank(val)) {
                    pd.set(ci, val);
                }
                break;
            case CI:
                String ref = ciReader.propertyAsCiRef(propNameInManifest);
                if (!isBlank(ref)) {
                    getReferences(context).add(new CiReference(ci, pd, ref));
                }
                break;
            case SET_OF_STRING:
            case LIST_OF_STRING:
                List<String> strings = ciReader.propertyAsStringCollection(propNameInManifest);
                if (!strings.isEmpty()) {
                    pd.set(ci, pd.getKind() == PropertyKind.SET_OF_STRING ? newHashSet(strings) : strings);
                }
                break;
            case SET_OF_CI:
            case LIST_OF_CI:
                if (pd.isAsContainment()) {
                    List<ConfigurationItem> items = readNestedCis(ciReader, propNameInManifest, ci, context, packageInfo);
                    if (items.size() != items.stream().map(ConfigurationItem::getName).distinct().count()) {
                        throw new ImporterException("Found 2 ci with same name while importing package , ensure there is only one.");
                    }
                    pd.set(ci, pd.getKind() == PropertyKind.SET_OF_CI ? newHashSet(items) : items);
                } else {
                    List<String> refs = ciReader.propertyAsCiRefCollection(propNameInManifest);
                    if (!refs.isEmpty()) {
                        getReferences(context).add(new CiReference(ci, pd, refs));
                    }
                }
                break;
            case MAP_STRING_STRING:
                Map<String, String> map = ciReader.propertyAsMapStringString(propNameInManifest);
                pd.set(ci, map);
                break;
        }
    }

    private List<ConfigurationItem> readNestedCis(
            ManifestReader.ManifestCiReader ciReader, String name, ConfigurationItem version, ImportingContext context, PackageInfo packageInfo) {

        List<ManifestReader.ManifestCiReader> readers = ciReader.propertyAsNestedCis(name);
        List<ConfigurationItem> configurationItems = newArrayList();
        List<String> undefinedProperties = new ArrayList<>();
        for (ManifestReader.ManifestCiReader manifestCiReader : readers) {
            ConfigurationItem ci = readCi(manifestCiReader, version.getId(), context, packageInfo);
            undefinedProperties.addAll(manifestCiReader.allProperties()
                    .stream()
                    .filter(manifestProperty -> !ci.hasProperty(manifestProperty))
                    .map(manifestProperty -> ci.getType().toString() + "." + manifestProperty)
                    .collect(toList()));
            configurationItems.add(ci);
        }

        if (!undefinedProperties.isEmpty()) {
            String undefinedPropertiesStr = undefinedProperties.stream().collect(joining(", "));
            String packageName = packageInfo.getApplicationName() + "/" + packageInfo.getApplicationVersion();
            if (version.getProperty("ignoreUndefinedPropertiesInManifest")) {
                logger.warn("While importing package {}, the following undefined properties were used in the manifest: {}.", packageName, undefinedPropertiesStr);
            } else {
                throw new ImporterException("While importing package [%s], the following undefined properties were used in the manifest: [%s].", packageName, undefinedPropertiesStr);
            }
        }
        return configurationItems;
    }

    private ConfigurationItem readCi(ManifestReader.ManifestCiReader ciReader, String parentId, ImportingContext context, PackageInfo packageInfo) {
        Type type = ciReader.type();
        ConfigurationItem configurationItem = type.getDescriptor().newInstance(generateId(parentId, ciReader.name()));

        type.getDescriptor()
                .getPropertyDescriptors()
                .stream()
                .filter(propertyDescriptor -> !propertyDescriptor.isHidden())
                .forEach(propertyDescriptor -> readProperty(ciReader, propertyDescriptor, configurationItem, context, packageInfo));

        if (configurationItem instanceof SourceArtifact) {
            String file = ciReader.file();
            boolean hasFileUriSet = ciReader.hasProperty(SourceArtifact.FILE_URI_PROPERTY_NAME);
            if (file != null) {
                if (hasFileUriSet) {
                    throw new ImporterException("Artifact %s has both file '%s' set, as well as file URI '%s'", ciReader.name(), file, ciReader.propertyAsString(SourceArtifact.FILE_URI_PROPERTY_NAME));
                }
                readArtifactData((Artifact) configurationItem, file, context);
            } else if (!hasFileUriSet) {
                throw new ImporterException("Artifact %s should have either 'file' set, or have a '%s' property", ciReader.name(), SourceArtifact.FILE_URI_PROPERTY_NAME);
            } else {
                String fileUri = ((SourceArtifact) configurationItem).getFileUri();

                try {
                    ArtifactResolver resolver = ArtifactResolverRegistry.lookup(fileUri);
                    if (!resolver.validateCorrectness((SourceArtifact) configurationItem)) {
                        throw new ImporterException("Artifact %s property %s with value %s failed to validate against its ArtifactResolver.", ciReader.name(), SourceArtifact.FILE_URI_PROPERTY_NAME, fileUri);
                    }
                } catch (IllegalArgumentException ex) {
                    throw new ImporterException(ex);
                }

            }
        }

        getReadCis(context).put(ciReader.name(), configurationItem);
        return configurationItem;
    }


    private void readArtifactData(Artifact artifact, String fileName, ImportingContext ctx) {
        // Need to wrap in TFile to detect this archivedetector
        TFile sourceArchive = new TFile(source.getFile());
        try {
            TFile tempFolder = createTempFolderForImport(ctx, artifact.getName());
            ctx.<List<TFile>>getAttribute(TEMPORARY_FILES).add(tempFolder);
            // Prevent going deep in archives by setting archivedetector to NULL.
            TFile dest = copyArtifactData(sourceArchive, fileName, tempFolder, artifact);
            artifact.setFile(LocalFile.valueOf(dest.getFile()));
        } finally {
            if (sourceArchive.isArchive()) {
                try {
                    TVFS.umount(sourceArchive);
                } catch (FsSyncException e) {
                    logger.error("Unable to release resources for archive {}", sourceArchive.getName());
                    logger.error("Following exception occurred while trying to release resources: {}", e.getMessage());
                }
            }
        }
    }

    private static TFile copyArtifactData(TFile sourceArchive, String fileName, TFile tempFolder, Artifact artifact) {
        TFile artifactFile = new TFile(sourceArchive, fileName, TArchiveDetector.NULL);
        if (!artifactFile.exists()) {
            throw new RuntimeIOException(format("The file %s could not be found in the package.", fileName));
        }

        TFile dest = new TFile(tempFolder, artifactFile.getName());

        try {
            artifactFile.cp_r(dest);
            TVFS.umount(dest);
        } catch (IOException e) {
            throw new RuntimeIOException(format("Could not copy %s to %s while importing %s", artifactFile, dest, artifact.getId()), e);
        }

        return dest;
    }

    private static TFile createTempFolderForImport(ImportingContext ctx, String artifactName) {
        try {
            String name = "temp-" + artifactName;
            TFile tempFolder = new TFile(File.createTempFile(name, "")).rm().mkdir(false);
            ctx.<List<TFile>>getAttribute(TEMPORARY_FILES).add(tempFolder);
            logger.debug("Created Temporary folder {}", tempFolder);
            return tempFolder;
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }



    static List<CiReference> getReferences(ImportingContext context) {
        List<CiReference> l = context.getAttribute(REFERENCES);
        if (l == null) {
            l = newArrayList();
            context.setAttribute(REFERENCES, l);
        }
        return l;
    }

    static Map<String, ConfigurationItem> getReadCis(ImportingContext context) {
        Map<String, ConfigurationItem> m = context.getAttribute(READ_CIS);
        if (m == null) {
            m = newHashMap();
            context.setAttribute(READ_CIS, m);
        }
        return m;
    }

    public void cleanUp(PackageInfo packageInfo, ImportingContext context) {
        List<TFile> files = context.getAttribute(TEMPORARY_FILES);
        for (TFile file : files) {
            logger.debug("Cleaning up temporary file {}", file);
            try {
                File javaFile = file.getFile();
                if (javaFile.isDirectory()) {
                    GuavaFiles.deleteRecursively(javaFile);
                } else {
                    if (javaFile.exists() && !javaFile.delete()) {
                        logger.info("Couldn't delete file: {}", javaFile);
                    }
                }
            } catch (IOException e) {
                logger.error("Couldn't clean up file {}", file, e);
            }
        }
        TFiles.umountQuietly(new TFile(packageInfo.getSource().getFile()));
    }

    private static final Logger logger = LoggerFactory.getLogger(ManifestReaderDriver.class);

    public void resolveReferences(RepositoryService repositoryService, ImportingContext ctx) {
        List<CiReference> references = getReferences(ctx);
        for (CiReference reference : references) {
            // Resolve IDs to CIs
            List<ConfigurationItem> resolvedCIs = newArrayList();
            for (String id : reference.getIds()) {
                ConfigurationItem alsoRead = getReadCis(ctx).get(id);
                if (alsoRead != null) {
                    resolvedCIs.add(alsoRead);
                } else {
                    resolvedCIs.add(repositoryService.read(id));
                }
            }

            // Set referred CIs on the parsed CI
            reference.set(resolvedCIs);
        }
    }
}
